#!/usr/bin/env python3
import gi
import subprocess
import threading
import os

gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, GLib

# ========== CONFIGURAZIONE ==========

MIRRORS = [
    'https://ufficiozero.mirror.garr.it/3rd-party-software/arubasign/24.0.2/ArubaSign-latest-LINUX.tar.bz2',
    'https://sourceforge.net/projects/ufficiozero/files/3rd-party-software/arubasign/24.0.2/ArubaSign-latest-LINUX.tar.bz2',
    'https://ftp.acc.umu.se/mirror/ufficiozero.org/3rd-party-software/arubasign/24.0.2/ArubaSign-latest-LINUX.tar.bz2',
    'https://ftp.icm.edu.pl/pub/Linux/dist/ufficiozero/3rd-party-software/arubasign/24.0.2/ArubaSign-latest-LINUX.tar.bz2',
    'https://mirrors.dotsrc.org/ufficiozero/3rd-party-software/arubasign/24.0.2/ArubaSign-latest-LINUX.tar.bz2',
    'https://ftp.nluug.nl/pub/os/Linux/distr/ufficiozero/3rd-party-software/arubasign/24.0.2/ArubaSign-latest-LINUX.tar.bz2',
    'https://quantum-mirror.hu/mirrors/pub/ufficiozero/3rd-party-software/arubasign/24.0.2/ArubaSign-latest-LINUX.tar.bz2'
]

INSTALL_DIR = os.path.expanduser("~/.arubasign")
ARCHIVE = "ArubaSign-latest-LINUX.tar.bz2"

# =====================================


class Installer(Gtk.Window):
    def __init__(self):
        super().__init__(title="Installazione ArubaSign")
        
        self.set_default_size(450, 140)
        self.set_position(Gtk.WindowPosition.CENTER)
        self.set_resizable(False)
        self.set_border_width(15)

        self.label = Gtk.Label(label="Preparazione installazione…")
        self.label.set_xalign(0)
        
        self.progress = Gtk.ProgressBar()
        self.progress.set_show_text(True)

        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
        box.pack_start(self.label, False, False, 0)
        box.pack_start(self.progress, False, False, 0)
        
        self.add(box)

        threading.Thread(target=self.run_install, daemon=True).start()

    def update_ui(self, text, fraction):
        self.label.set_text(text)
        self.progress.set_fraction(fraction)
        if fraction < 1.0:
            self.progress.set_text(f"{int(fraction * 100)}%")
        else:
            self.progress.set_text("Completato")
        return False

    def show_error(self, message):
        dialog = Gtk.MessageDialog(
            transient_for=self,
            flags=0,
            message_type=Gtk.MessageType.ERROR,
            buttons=Gtk.ButtonsType.OK,
            text="Errore di installazione"
        )
        dialog.format_secondary_text(message)
        dialog.run()
        dialog.destroy()
        Gtk.main_quit()
        return False

    def run_command(self, cmd, cwd=None):
        result = subprocess.run(
            cmd,
            shell=True,
            cwd=cwd,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        )
        return result.returncode == 0

    def run_install(self):
        try:
            GLib.idle_add(self.update_ui, "Preparazione ambiente…", 0.05)
            os.makedirs(INSTALL_DIR, exist_ok=True)

            GLib.idle_add(self.update_ui, "Download ArubaSign…", 0.10)
            
            archive_path = os.path.join(INSTALL_DIR, ARCHIVE)
            downloaded = False
            
            for idx, mirror in enumerate(MIRRORS, 1):
                GLib.idle_add(self.update_ui, f"Download da mirror {idx}/{len(MIRRORS)}…", 0.10 + (idx * 0.05))
                
                if self.run_command(f"wget -q --timeout=15 -O '{archive_path}' '{mirror}'"):
                    downloaded = True
                    break
            
            if not downloaded:
                GLib.idle_add(self.show_error, "Impossibile scaricare ArubaSign da nessun mirror.")
                return

            GLib.idle_add(self.update_ui, "Estrazione archivio…", 0.50)
            if not self.run_command(f"tar -xjvf '{archive_path}' -C '{INSTALL_DIR}'"):
                GLib.idle_add(self.show_error, "Errore durante l'estrazione.")
                return

            GLib.idle_add(self.update_ui, "Configurazione…", 0.70)
            
            extracted_dirs = [d for d in os.listdir(INSTALL_DIR) 
                            if os.path.isdir(os.path.join(INSTALL_DIR, d))]
            
            if not extracted_dirs:
                GLib.idle_add(self.show_error, f"Directory ArubaSign non trovata in {INSTALL_DIR}")
                return
            
            aruba_dir = sorted(extracted_dirs)[-1]
            install_script = os.path.join(INSTALL_DIR, aruba_dir, "install.sh")

            if not os.path.exists(install_script):
                GLib.idle_add(self.show_error, f"Script di installazione non trovato:\n{install_script}")
                return

            GLib.idle_add(self.update_ui, "Installazione in corso…", 0.80)
            
            os.chmod(install_script, 0o755)
            
            if not self.run_command(f"'{install_script}'", cwd=os.path.dirname(install_script)):
                GLib.idle_add(self.show_error, "Errore durante l'installazione di ArubaSign.")
                return

            GLib.idle_add(self.update_ui, "Pulizia file temporanei…", 0.95)
            if os.path.exists(archive_path):
                os.remove(archive_path)

            GLib.idle_add(self.update_ui, "✓ Installazione completata con successo", 1.0)
            GLib.timeout_add_seconds(3, Gtk.main_quit)

        except Exception as e:
            GLib.idle_add(self.show_error, f"Errore imprevisto:\n{str(e)}")


if __name__ == "__main__":
    win = Installer()
    win.connect("destroy", Gtk.main_quit)
    win.show_all()
    Gtk.main()
