#!/usr/bin/env python3
import gi
gi.require_version('Gtk', '3.0')
from gi.repository import Gtk, Gdk, GLib, Gio, GdkPixbuf
import subprocess
import threading
import os
from datetime import datetime
import json
import sys
import traceback
import re
import sqlite3

# --- Import Classes from other files ---
from repositories import RepositoriesDialog
from description import InfoDialog

# --- Global Definitions ---

CUSTOM_CSS = b"""
.button-border {
  border: 1px solid rgba(0, 0, 0, 0.2);
}
"""

CONFIG_DIR = os.path.join(os.path.expanduser("~"), ".config", "dnf5-frontend")
CONFIG_FILE = os.path.join(CONFIG_DIR, "preferences.json")
DNF_CMD = "dnf"
HISTORY_PATH = "/usr/lib/sysimage/libdnf5/transaction_history.sqlite"

def _detect_dnf_command():
    """Detects if dnf5 is available and returns the correct command."""
    if os.path.exists("/usr/bin/dnf5"):
        return "dnf5"
    else:
        return "dnf"

# --- Main Application Class (with all functionality merged) ---

class PackageApp(Gtk.ApplicationWindow):
    def __init__(self, app):
        Gtk.ApplicationWindow.__init__(self, application=app, title="DNF Package Manager")

        self.preferences = self.load_preferences()
        self.dnf_cmd = self.preferences.get("dnf_cmd", DNF_CMD)

        self.upgradable_liststore = Gtk.ListStore(bool, str, str, str, str)
        self.installed_liststore = Gtk.ListStore(bool, str, str, str, str)
        self.available_packages_liststore = Gtk.ListStore(bool, str, str, str, str)

        self.upgradable_packages_set = set()
        self.installed_packages_set = set()

        self.upgradable_filter = self.upgradable_liststore.filter_new()
        self.upgradable_filter.set_visible_func(self.filter_packages)
        self.installed_filter = self.installed_liststore.filter_new()
        self.installed_filter.set_visible_func(self.filter_packages)
        self.available_packages_filter = self.available_packages_liststore.filter_new()
        self.available_packages_filter.set_visible_func(self.filter_packages)

        self.status_bar_map = {
            0: ("🟠️ Upgradable", self.upgradable_filter, self.upgradable_liststore),
            1: ("🟢️ Installed", self.installed_filter, self.installed_liststore),
            2: ("🔵️ Available", self.available_packages_filter, self.available_packages_liststore),
        }

        self.last_paned_position = 0

        window_width = self.preferences.get("window_width", 1000)
        window_height = self.preferences.get("window_height", 700)
        self.set_default_size(window_width, window_height)

        self.main_paned = Gtk.Paned(orientation=Gtk.Orientation.VERTICAL)
        self.add(self.main_paned)

        self.vbox = Gtk.VBox(spacing=6)
        self.vbox.set_border_width(6)
        self.main_paned.pack1(self.vbox, True, False)

        menubar = Gtk.MenuBar()
        self.vbox.pack_start(menubar, False, False, 0)
        
        preferences_item = Gtk.MenuItem.new_with_mnemonic("✴️ _Preferences")
        preferences_item.connect("activate", self.on_preferences_button_clicked)
        preferences_item.connect("button-press-event", self.on_menu_item_button_press)
        menubar.append(preferences_item)
        
        # New History Menu Item
        history_item = Gtk.MenuItem.new_with_mnemonic("📖 _History")
        history_item.connect("activate", self.on_history_button_clicked)
        history_item.connect("button-press-event", self.on_menu_item_button_press)
        menubar.append(history_item)

        about_item = Gtk.MenuItem.new_with_mnemonic("⭐️ _About")
        about_item.connect("activate", self.on_about_button_clicked)
        about_item.connect("button-press-event", self.on_menu_item_button_press)
        menubar.append(about_item)
        
        header_bar = Gtk.HeaderBar()
        header_bar.set_show_close_button(True)
        header_bar.set_title("DNF Package Manager")
        self.set_titlebar(header_bar)
        
        toolbar = Gtk.Toolbar()
        self.vbox.pack_start(toolbar, False, False, 0)
        
        self.search_entry = Gtk.SearchEntry()
        self.search_entry.set_placeholder_text("Search packages...")
        self.search_entry.connect("search-changed", self.on_search_changed)
        
        self.search_entry.set_size_request(150, -1)

        tool_item_search = Gtk.ToolItem()
        tool_item_search.set_expand(True)
        tool_item_search.add(self.search_entry)
        toolbar.add(tool_item_search)

        self.spinner = Gtk.Spinner()
        self.spinner.set_tooltip_text("🕓️ Operation in progress")
        tool_item_spinner = Gtk.ToolItem()
        tool_item_spinner.add(self.spinner)
        toolbar.add(tool_item_spinner)
        self.spinner.hide()

        package_list_action_bar = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6)
        package_list_action_bar.set_border_width(6)
        self.vbox.pack_start(package_list_action_bar, False, False, 0)

        select_all_button = Gtk.Button.new_from_icon_name("", Gtk.IconSize.MENU)
        select_all_button.set_label("✅️ Select All  ")
        select_all_button.set_tooltip_text("✅️ Select/Deselect all visible packages")
        select_all_button.connect("clicked", self.on_select_all_button_clicked)
        package_list_action_bar.pack_start(select_all_button, False, False, 0)

        refresh_button = Gtk.Button.new_from_icon_name("", Gtk.IconSize.MENU)
        refresh_button.set_label("♻️ Refresh  ")
        refresh_button.set_tooltip_text("♻️ Refresh package lists")
        refresh_button.connect("clicked", self.on_load_button_clicked)
        package_list_action_bar.pack_start(refresh_button, False, False, 0)

        repositories_button = Gtk.Button()
        repositories_button.set_label("®️ Repositories")
        repositories_button.set_tooltip_text("®️ Manage repositories")
        repositories_button.connect("clicked", self.on_repositories_button_clicked)
        package_list_action_bar.pack_start(repositories_button, False, False, 0)

        pin_unpin_button = Gtk.Button()
        pin_unpin_button.set_label("📍 Pin/Unpin")
        pin_unpin_button.set_tooltip_text("📍 Manage pinned packages")
        pin_unpin_button.connect("clicked", self.on_pin_unpin_button_clicked)
        package_list_action_bar.pack_start(pin_unpin_button, False, False, 0)

        package_list_action_bar.pack_start(Gtk.Label(), True, True, 0)

        self.statusbar_label = Gtk.Label(label="Ready.")
        package_list_action_bar.pack_end(self.statusbar_label, False, False, 0)

        self.notebook = Gtk.Notebook()
        self.vbox.pack_start(self.notebook, True, True, 0)
        self.notebook.connect("switch-page", self.on_notebook_page_changed)

        self.notebook.handler_block_by_func(self.on_notebook_page_changed)

        upgradable_page = self.create_package_page(self.upgradable_filter, "Upgradable")
        self.notebook.append_page(upgradable_page, Gtk.Label(label="🟠️ Upgradable"))

        installed_page = self.create_package_page(self.installed_filter, "Installed")
        self.notebook.append_page(installed_page, Gtk.Label(label="🟢️ Installed"))

        available_packages_page = self.create_package_page(self.available_packages_filter, "Available")
        self.notebook.append_page(available_packages_page, Gtk.Label(label="🔵️ Available"))
        
        self.notebook.handler_unblock_by_func(self.on_notebook_page_changed)

        self.action_bar = Gtk.ActionBar()
        self.vbox.pack_start(self.action_bar, False, False, 0)

        self.install_button = Gtk.Button.new_from_icon_name("", Gtk.IconSize.BUTTON)
        self.install_button.set_label("🟢️ Install ")
        self.install_button.connect("clicked", self.on_install_button_clicked)

        self.remove_button = Gtk.Button.new_from_icon_name("", Gtk.IconSize.BUTTON)
        self.remove_button.set_label("🔴️ Remove  ")
        self.remove_button.connect("clicked", self.on_remove_button_clicked)

        self.update_button = Gtk.Button.new_from_icon_name("", Gtk.IconSize.BUTTON)
        self.update_button.set_label("🟠️ Update  ")
        self.update_button.connect("clicked", self.on_update_button_clicked)

        self.reinstall_button = Gtk.Button.new_from_icon_name("", Gtk.IconSize.BUTTON)
        self.reinstall_button.set_label("♻️ Reinstall ")
        self.reinstall_button.connect("clicked", self.on_reinstall_button_clicked)

        self.description_button = Gtk.Button.new_from_icon_name("", Gtk.IconSize.BUTTON)
        self.description_button.set_label("⚪️ Description  ")
        self.description_button.connect("clicked", self.on_properties_button_clicked)

        self.action_bar.pack_start(self.install_button)
        self.action_bar.pack_start(self.remove_button)
        self.action_bar.pack_start(self.update_button)
        self.action_bar.pack_start(self.reinstall_button)
        self.action_bar.pack_start(self.description_button)

        self.log_toggle_button = Gtk.ToggleButton()
        self.log_arrow = Gtk.Arrow(arrow_type=Gtk.ArrowType.DOWN, shadow_type=Gtk.ShadowType.OUT)
        self.log_toggle_button.add(self.log_arrow)
        self.log_toggle_button.set_tooltip_text("Show/hide logs")
        self.log_toggle_button.connect("toggled", self.on_log_toggled)
        self.log_toggle_button.set_label("⚫️ View Logs")
        self.action_bar.pack_end(self.log_toggle_button)

        self.log_textview = Gtk.TextView()
        self.log_textview.set_editable(False)
        self.log_textview.set_cursor_visible(False)
        self.log_textview.set_wrap_mode(Gtk.WrapMode.WORD)
        log_scroll = Gtk.ScrolledWindow()
        log_scroll.add(self.log_textview)
        
        self.log_window = Gtk.Window(type=Gtk.WindowType.POPUP)
        self.log_window.set_transient_for(self)
        self.log_window.set_decorated(False)
        self.log_window.set_default_size(-1, 150)
        self.log_window.set_resizable(True)
        self.log_window.add(log_scroll)
        self.log_window.connect("size-allocate", self.on_log_window_size_allocate)
        self.log_window.hide()

        self.connect("realize", self.on_window_realize)
        self.apply_custom_css()

        if not self.load_data_from_cache():
            self.on_load_button_clicked()
        else:
            self._update_gui_from_cache()

        self.connect('delete-event', self.on_quit_clicked)
    
    # --- START OF PinDialog CLASS ---
    class PinDialog(Gtk.Dialog):
        def __init__(self, parent, dnf_cmd, log_func, run_cmd_func):
            super().__init__(
                title="Manage Package Pin/Unpin",
                parent=parent,
                flags=0,
                buttons=(Gtk.STOCK_CLOSE, Gtk.ResponseType.CLOSE)
            )
            self.dnf_cmd = dnf_cmd
            self.log = log_func
            self.run_command_in_thread = run_cmd_func
            self.parent = parent
            self.connect("response", self.on_response)
            self.set_default_size(600, 400)
            
            content_area = self.get_content_area()
            self.notebook = Gtk.Notebook()
            content_area.add(self.notebook)

            pin_tab = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6, margin=10)
            self.notebook.append_page(pin_tab, Gtk.Label(label="Pin a Package"))

            pin_search_entry = Gtk.SearchEntry()
            pin_search_entry.set_placeholder_text("Search installed packages...")
            pin_tab.pack_start(pin_search_entry, False, False, 0)

            self.pin_liststore = Gtk.ListStore(str)
            self.pin_filter = self.pin_liststore.filter_new()
            self.pin_filter.set_visible_func(self.filter_pin_list)
            
            pin_search_entry.connect("search-changed", self.on_pin_search_changed)

            self.pin_treeview = Gtk.TreeView(model=self.pin_filter)
            renderer_text = Gtk.CellRendererText()
            column = Gtk.TreeViewColumn("Package Name", renderer_text, text=0)
            self.pin_treeview.append_column(column)

            scrolled_window_pin = Gtk.ScrolledWindow()
            scrolled_window_pin.set_shadow_type(Gtk.ShadowType.IN)
            scrolled_window_pin.add(self.pin_treeview)
            pin_tab.pack_start(scrolled_window_pin, True, True, 0)
            
            pin_button = Gtk.Button.new_with_label("Pin Selected Package")
            pin_button.connect("clicked", self.on_pin_clicked)
            pin_tab.pack_end(pin_button, False, False, 0)
            self.pin_selection = self.pin_treeview.get_selection()
            self.pin_selection.set_mode(Gtk.SelectionMode.SINGLE)
            
            self.populate_installed_packages()

            unpin_tab = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6, margin=10)
            self.notebook.append_page(unpin_tab, Gtk.Label(label="Unpin Packages"))

            self.unpin_liststore = Gtk.ListStore(str)
            self.unpin_treeview = Gtk.TreeView(model=self.unpin_liststore)

            renderer = Gtk.CellRendererText()
            column = Gtk.TreeViewColumn("Package Name", renderer, text=0)
            self.unpin_treeview.append_column(column)
            
            scrolled_window_unpin = Gtk.ScrolledWindow()
            scrolled_window_unpin.set_shadow_type(Gtk.ShadowType.IN)
            scrolled_window_unpin.add(self.unpin_treeview)
            unpin_tab.pack_start(scrolled_window_unpin, True, True, 0)

            unpin_button = Gtk.Button.new_with_label("Unpin Selected")
            unpin_button.connect("clicked", self.on_unpin_clicked)
            unpin_button.set_sensitive(False)
            unpin_tab.pack_end(unpin_button, False, False, 0)

            unpin_selection = self.unpin_treeview.get_selection()
            unpin_selection.connect("changed", lambda s: unpin_button.set_sensitive(s.get_selected()[1] is not None))
            
            self.populate_pins()
            
            self.show_all()

        def on_response(self, dialog, response_id):
            if response_id == Gtk.ResponseType.CLOSE:
                dialog.destroy()

        def on_pin_search_changed(self, entry):
            self.pin_filter.refilter()

        def filter_pin_list(self, model, iter, data):
            search_text = self.notebook.get_nth_page(0).get_children()[0].get_text().lower()
            if not search_text:
                return True
            package_name = model.get_value(iter, 0).lower()
            return search_text in package_name

        def populate_installed_packages(self):
            self.pin_liststore.clear()
            self.log("Fetching list of installed packages for pinning...")
            self.run_command_in_thread(
                [self.dnf_cmd, "repoquery", "--installed", "--queryformat", "%{name}\n"],
                "", "",
                on_finish_callback=self._on_installed_list_complete
            )
            
        def _on_installed_list_complete(self, stdout, stderr, returncode):
            if returncode != 0:
                self.log(f"Failed to list installed packages: {stderr.strip()}")
                return
            for line in stdout.splitlines():
                pkg_name = line.strip()
                if pkg_name:
                    self.pin_liststore.append([pkg_name])
            self.log(f"Found {len(self.pin_liststore)} installed packages.")

        def populate_pins(self):
            self.unpin_liststore.clear()
            self.log("Querying pinned packages...")
            self.run_command_in_thread(
                [self.dnf_cmd, "versionlock", "list"],
                "", "",
                on_finish_callback=self._on_unpin_list_complete
            )

        def _on_unpin_list_complete(self, stdout, stderr, returncode):
            if returncode != 0:
                self.log(f"Failed to list pinned packages: {stderr.strip()}")
                return

            lines = stdout.splitlines()
            i = 0
            while i < len(lines):
                line = lines[i].strip()
                
                if not line or line.startswith("#"):
                    i += 1
                    continue
                
                if line.startswith("Package name:"):
                    pkg_name = line.replace("Package name:", "").strip()
                    self.unpin_liststore.append([pkg_name])
                    
                    i += 2
                else:
                    i += 1

            self.log(f"Found {len(self.unpin_liststore)} pinned packages.")

        def on_pin_clicked(self, widget):
            model, treeiter = self.pin_selection.get_selected()
            if not treeiter:
                self.log("Please select a package to pin.")
                return

            pkg_name = model[treeiter][0]
            self.run_command_in_thread(
                [self.dnf_cmd, "versionlock", "add", pkg_name],
                f"Pinned {pkg_name}.",
                f"Failed to pin {pkg_name}.",
                on_finish_callback=lambda s, e, r: self.populate_pins(),
                use_pkexec=True
            )

        def on_unpin_clicked(self, widget):
            selection = self.unpin_treeview.get_selection()
            model, treeiter = selection.get_selected()
            if treeiter:
                pkg_name = model[treeiter][0]
                
                self.run_command_in_thread(
                    [self.dnf_cmd, "versionlock", "delete", pkg_name],
                    f"Unpinned {pkg_name} successfully.",
                    f"Failed to unpin {pkg_name}.",
                    on_finish_callback=lambda s, e, r: self.populate_pins(),
                    use_pkexec=True
                )
    
    # --- END OF PinDialog CLASS ---
    # --- Updated PreferencesWindow Class (nested) ---
    class PreferencesWindow(Gtk.Dialog):
        def __init__(self, parent, preferences):
            super().__init__(
                title="Preferences",
                parent=parent,
                flags=0,
                buttons=(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
                         Gtk.STOCK_APPLY, Gtk.ResponseType.APPLY,
                         Gtk.STOCK_CLOSE, Gtk.ResponseType.CLOSE)
            )
            self.parent = parent
            self.preferences = preferences.copy()
            self.set_default_size(400, 300)

            grid = Gtk.Grid()
            grid.set_column_spacing(10)
            grid.set_row_spacing(10)
            grid.set_border_width(12)
            self.get_content_area().add(grid)

            row = 0

            general_frame = Gtk.Frame(label="General Options")
            grid.attach(general_frame, 0, row, 2, 1)
            general_grid = Gtk.Grid()
            general_grid.set_column_spacing(10)
            general_grid.set_row_spacing(10)
            general_grid.set_border_width(12)
            general_frame.add(general_grid)

            self.disable_confirm_check = Gtk.CheckButton.new_with_label("Disable confirmation dialogs for actions")
            self.disable_confirm_check.set_active(self.preferences.get("disable_confirm", False))
            general_grid.attach(self.disable_confirm_check, 0, 0, 2, 1)

            self.clear_cache_check = Gtk.CheckButton.new_with_label("Clean DNF cache after successful action")
            self.clear_cache_check.set_active(self.preferences.get("clear_cache", False))
            general_grid.attach(self.clear_cache_check, 0, 1, 2, 1)
            
            row += 1

            history_frame = Gtk.Frame(label="History Options")
            grid.attach(history_frame, 0, row, 2, 1)
            history_grid = Gtk.Grid()
            history_grid.set_column_spacing(10)
            history_grid.set_row_spacing(10)
            history_grid.set_border_width(12)
            history_frame.add(history_grid)

            history_size_label = Gtk.Label(label="History Size:", xalign=0)
            history_grid.attach(history_size_label, 0, 0, 1, 1)
            
            self.history_size_adj = Gtk.Adjustment(
                value=self.preferences.get("history_size", 50),
                lower=1,
                upper=1000,
                step_increment=1,
                page_increment=10
            )
            self.history_size_spin = Gtk.SpinButton(
                adjustment=self.history_size_adj,
                climb_rate=0,
                digits=0
            )
            history_grid.attach(self.history_size_spin, 1, 0, 1, 1)

            self.connect("response", self.on_response)
            self.show_all()

        def on_response(self, dialog, response_id):
            if response_id in (Gtk.ResponseType.APPLY, Gtk.ResponseType.CLOSE):
                self.preferences["disable_confirm"] = self.disable_confirm_check.get_active()
                self.preferences["clear_cache"] = self.clear_cache_check.get_active()
                self.preferences["history_size"] = self.history_size_spin.get_value_as_int()

                self.parent.preferences.update(self.preferences)
                self.parent.save_preferences(self.parent.preferences)

                if response_id == Gtk.ResponseType.APPLY:
                    self.parent.log("Preferences applied.")
                elif response_id == Gtk.ResponseType.CLOSE:
                    self.parent.log("Preferences saved on close.")
            
            if response_id == Gtk.ResponseType.CANCEL or response_id == Gtk.ResponseType.CLOSE:
                dialog.destroy()

    # --- Corrected HistoryDialog Class (nested) ---
    class HistoryDialog(Gtk.Dialog):
        def __init__(self, parent):
            super().__init__(
                title="DNF Transaction History",
                parent=parent,
                flags=0
            )
            self.parent = parent
            self.set_default_size(600, 450)

            self.add_buttons(
                Gtk.STOCK_CLOSE, Gtk.ResponseType.CLOSE
            )
            
            content_area = self.get_content_area()
            vbox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6, margin=10)
            content_area.add(vbox)

            self.liststore = Gtk.ListStore(int, str, str, str)
            self.treeview = Gtk.TreeView(model=self.liststore)
            self.treeview.set_vexpand(True)

            columns = ["Transaction ID", "Date", "Time", "Description"]
            for i, col_name in enumerate(columns):
                renderer = Gtk.CellRendererText()
                column = Gtk.TreeViewColumn(col_name, renderer, text=i)
                self.treeview.append_column(column)
            
            scrolled_window = Gtk.ScrolledWindow()
            scrolled_window.set_shadow_type(Gtk.ShadowType.IN)
            scrolled_window.add(self.treeview)
            vbox.pack_start(scrolled_window, True, True, 0)

            self.details_textview = Gtk.TextView()
            self.details_textview.set_editable(False)
            self.details_textview.set_cursor_visible(False)
            self.details_textview.set_wrap_mode(Gtk.WrapMode.WORD)
            details_scroll = Gtk.ScrolledWindow()
            details_scroll.set_size_request(-1, 200)
            details_scroll.add(self.details_textview)
            vbox.pack_start(details_scroll, False, False, 0)
            
            action_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6, margin_top=6)
            vbox.pack_start(action_box, False, False, 0)

            self.delete_history_button = Gtk.Button.new_with_label("⚠️ Delete All History")
            self.delete_history_button.connect("clicked", self.on_delete_history_clicked)
            action_box.pack_end(self.delete_history_button, False, False, 0)
            
            self.revert_button = Gtk.Button.new_with_label("⏪ Revert Transaction")
            self.revert_button.connect("clicked", self.on_revert_clicked)
            action_box.pack_start(self.revert_button, False, False, 0)
            self.revert_button.set_sensitive(False)

            self.save_button = Gtk.Button.new_with_label("💾 Save")
            self.save_button.connect("clicked", self.on_save_clicked)
            action_box.pack_start(self.save_button, False, False, 0)
            self.save_button.set_sensitive(False)

            self.treeview.get_selection().connect("changed", self.on_selection_changed)
            self.connect("response", self.on_dialog_response)
            
            self.populate_history()
            self.show_all()

        def on_dialog_response(self, dialog, response_id):
            if response_id == Gtk.ResponseType.CLOSE:
                dialog.destroy()
                
        def on_selection_changed(self, selection):
            model, treeiter = selection.get_selected()
            if treeiter:
                self.revert_button.set_sensitive(True)
                self.save_button.set_sensitive(True)
                trans_id = model[treeiter][0]
                self.parent.log(f"Fetching details for transaction ID {trans_id}...")
                self.parent.run_command_in_thread(
                    [self.parent.dnf_cmd, "history", "info", str(trans_id)],
                    "", "",
                    on_finish_callback=self._on_details_complete
                )
            else:
                self.revert_button.set_sensitive(False)
                self.save_button.set_sensitive(False)
                self.details_textview.get_buffer().set_text("")
                
        def _on_details_complete(self, stdout, stderr, returncode):
            buffer = self.details_textview.get_buffer()
            buffer.set_text("")
            
            if returncode == 0:
                if stdout.strip():
                    buffer.set_text(stdout.strip())
                else:
                    buffer.set_text("No detailed information found for this transaction.")
            else:
                buffer.set_text(f"Error fetching details (Return code {returncode}):\n{stderr.strip()}")
                self.parent.log(f"Error fetching details: {stderr.strip()}")

        def populate_history(self):
            self.liststore.clear()
            self.parent.log("Connecting to DNF transaction history database...")

            HISTORY_PATH = "/usr/lib/sysimage/libdnf5/transaction_history.sqlite"
            if not os.path.exists(HISTORY_PATH):
                self.parent.log(f"Error: History database not found at {HISTORY_PATH}.")
                self.parent.log("The History feature requires dnf5 and a transaction history.")
                return

            try:
                conn = sqlite3.connect(HISTORY_PATH)
                cursor = conn.cursor()
                cursor.execute("SELECT id, description, dt_begin FROM trans ORDER BY id DESC")
                
                rows = cursor.fetchall()
                if not rows:
                    self.parent.log("No transaction history found.")
                
                for row in rows:
                    trans_id = row[0]
                    description = row[1]
                    start_time_unix = row[2]

                    start_datetime = datetime.fromtimestamp(start_time_unix)
                    date_str = start_datetime.strftime("%Y-%m-%d")
                    time_str = start_datetime.strftime("%H:%M:%S")

                    self.liststore.append([trans_id, date_str, time_str, description])
                
                conn.close()
                self.parent.log(f"Loaded {len(rows)} history entries.")

            except sqlite3.Error as e:
                self.parent.log(f"SQLite error: {e}")
                
        def on_delete_history_clicked(self, widget):
            dialog = Gtk.MessageDialog(
                parent=self,
                flags=0,
                message_type=Gtk.MessageType.WARNING,
                buttons=Gtk.ButtonsType.OK_CANCEL,
                text="Are you sure you want to delete the entire history?"
            )
            dialog.format_secondary_text("This action is irreversible and will permanently remove all DNF transaction records. Proceed with caution!")
            response = dialog.run()
            dialog.destroy()
            
            if response == Gtk.ResponseType.OK:
                HISTORY_PATH = "/usr/lib/sysimage/libdnf5/transaction_history.sqlite"
                if not os.path.exists(HISTORY_PATH):
                    self.parent.log(f"Error: History database not found at {HISTORY_PATH}.")
                    return

                try:
                    conn = sqlite3.connect(HISTORY_PATH)
                    cursor = conn.cursor()
                    cursor.execute("DELETE FROM trans")
                    conn.commit()
                    conn.close()
                    self.parent.log("DNF transaction history deleted successfully.")
                    self.populate_history()
                except sqlite3.Error as e:
                    self.parent.log(f"Error deleting history: {e}")
                    
        def on_revert_clicked(self, widget):
            selection = self.treeview.get_selection()
            model, treeiter = selection.get_selected()
            if treeiter:
                trans_id = model[treeiter][0]
                dialog = Gtk.MessageDialog(
                    parent=self,
                    flags=0,
                    message_type=Gtk.MessageType.QUESTION,
                    buttons=Gtk.ButtonsType.OK_CANCEL,
                    text=f"Are you sure you want to revert transaction ID {trans_id}?"
                )
                dialog.format_secondary_text("This will undo all package changes made in this transaction.")
                response = dialog.run()
                dialog.destroy()
                
                if response == Gtk.ResponseType.OK:
                    self.parent.log(f"Reverting transaction ID {trans_id}...")
                    self.parent.run_command_in_thread(
                        [self.parent.dnf_cmd, "history", "undo", str(trans_id)],
                        f"Transaction {trans_id} reverted successfully.",
                        f"Failed to revert transaction {trans_id}.",
                        on_finish_callback=lambda s,e,r: self.populate_history(),
                        use_pkexec=True
                    )

        def on_save_clicked(self, widget):
            text_buffer = self.details_textview.get_buffer()
            text_to_save = text_buffer.get_text(
                text_buffer.get_start_iter(),
                text_buffer.get_end_iter(),
                False
            )

            if not text_to_save.strip():
                dialog = Gtk.MessageDialog(
                    parent=self,
                    flags=0,
                    message_type=Gtk.MessageType.INFO,
                    buttons=Gtk.ButtonsType.OK,
                    text="Nothing to save."
                )
                dialog.format_secondary_text("Please select a transaction to display its details before saving.")
                dialog.run()
                dialog.destroy()
                return

            dialog = Gtk.FileChooserDialog(
                title="Save Transaction Details",
                parent=self,
                action=Gtk.FileChooserAction.SAVE,
                buttons=(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL, Gtk.STOCK_SAVE, Gtk.ResponseType.OK)
            )
            dialog.set_do_overwrite_confirmation(True)

            try:
                documents_dir = os.path.join(os.path.expanduser("~"), "Documents")
                if not os.path.exists(documents_dir):
                    os.makedirs(documents_dir)
                dialog.set_current_folder(documents_dir)
            except Exception as e:
                self.parent.log(f"Could not set default save directory: {e}")

            trans_id = self.treeview.get_model()[self.treeview.get_selection().get_selected()[1]][0]
            default_filename = f"dnf_history_trans_{trans_id}.log"
            dialog.set_current_name(default_filename)

            response = dialog.run()

            if response == Gtk.ResponseType.OK:
                file_path = dialog.get_filename()
                try:
                    with open(file_path, "w") as f:
                        f.write(text_to_save)
                    self.parent.log(f"Successfully saved transaction details to {file_path}")
                except Exception as e:
                    self.parent.log(f"Error saving file: {e}")
                    error_dialog = Gtk.MessageDialog(
                        parent=self,
                        flags=0,
                        message_type=Gtk.MessageType.ERROR,
                        buttons=Gtk.ButtonsType.OK,
                        text="Error saving file"
                    )
                    error_dialog.format_secondary_text(f"An error occurred while saving the file:\n{e}")
                    error_dialog.run()
                    error_dialog.destroy()
            
            dialog.destroy()

    def on_response(self, dialog, response_id):
        if response_id == Gtk.ResponseType.CLOSE:
            dialog.destroy()

    def on_menu_item_button_press(self, widget, event):
        if event.button == Gdk.BUTTON_PRIMARY:
            widget.activate()
            return True
        return False

    def on_quit_clicked(self, window, event):
        self.save_preferences(self.preferences)
        return False

    def on_window_realize(self, widget):
        self.main_paned.set_position(self.get_size()[1])

    def on_log_window_size_allocate(self, widget, allocation):
        main_x, main_y = self.get_position()
        main_width, main_height = self.get_size()
        log_width = main_width
        log_height = allocation.height

        self.log_window.set_size_request(log_width, log_height)
        self.log_window.move(main_x, main_y + main_height - log_height)

    def create_package_page(self, liststore_filter, label):
        vbox = Gtk.VBox(spacing=6)
        vbox.set_border_width(6)

        scrolled_window = Gtk.ScrolledWindow()
        scrolled_window.set_hexpand(True)
        scrolled_window.set_vexpand(True)

        treeview = Gtk.TreeView(model=liststore_filter)
        treeview.set_headers_visible(True)
        treeview.connect("row-activated", self.on_row_activated)
        treeview.connect("button-press-event", self.on_treeview_button_press_event)

        renderer_toggle = Gtk.CellRendererToggle()
        renderer_toggle.connect("toggled", self.on_toggled_package, liststore_filter)
        column_toggle = Gtk.TreeViewColumn("Select", renderer_toggle, active=0)
        treeview.append_column(column_toggle)

        renderer_text_name = Gtk.CellRendererText()
        column_name = Gtk.TreeViewColumn("Name", renderer_text_name, text=1)

        column_name.set_min_width(125)
        column_name.set_max_width(225)
        
        column_name.set_resizable(True)
        treeview.append_column(column_name)

        renderer_text_version = Gtk.CellRendererText()
        column_version = Gtk.TreeViewColumn("Version", renderer_text_version, text=2)
        column_version.set_min_width(80)    
        column_version.set_max_width(150)   
        column_version.set_expand(False)
        column_version.set_resizable(True)
        treeview.append_column(column_version)

        renderer_text_summary = Gtk.CellRendererText()
        column_summary = Gtk.TreeViewColumn("Summary", renderer_text_summary, text=3)
        column_summary.set_resizable(True)
        treeview.append_column(column_summary)

        renderer_text_status = Gtk.CellRendererText()
        column_status = Gtk.TreeViewColumn("Status", renderer_text_status, text=4)
        column_status.set_resizable(True)
        treeview.append_column(column_status)

        scrolled_window.add(treeview)
        vbox.pack_start(scrolled_window, True, True, 0)

        return vbox

    def on_treeview_button_press_event(self, treeview, event):
        if event.button == Gdk.BUTTON_SECONDARY:
            path, column, _, _ = treeview.get_path_at_pos(int(event.x), int(event.y))
            if path:
                treeview.grab_focus()
                treeview.set_cursor(path)

                model = treeview.get_model()
                treeiter = model.get_iter(path)
                package_name = model.get_value(treeiter, 1)

                menu = Gtk.Menu()
                
                properties_item = Gtk.ImageMenuItem.new_with_label("🔶️ Properties")
                properties_item.set_image(Gtk.Image.new_from_icon_name("", Gtk.IconSize.MENU))
                properties_item.connect("activate", self.on_properties_button_clicked, [package_name])
                menu.append(properties_item)
                
                dependencies_item = Gtk.ImageMenuItem.new_with_label("🔷️ Show Dependencies")
                dependencies_item.set_image(Gtk.Image.new_from_icon_name("", Gtk.IconSize.MENU))
                dependencies_item.connect("activate", self.on_dependencies_button_clicked, [package_name])
                menu.append(dependencies_item)

                page_num = self.notebook.get_current_page()
                if page_num == 0:
                    update_item = Gtk.ImageMenuItem.new_with_label("🟠️ Update")
                    update_item.set_image(Gtk.Image.new_from_icon_name("", Gtk.IconSize.MENU))
                    update_item.connect("activate", self.on_update_button_clicked, [package_name])
                    menu.append(update_item)
                elif page_num == 1:
                    uninstall_item = Gtk.ImageMenuItem.new_with_label("🔴️ Uninstall")
                    uninstall_item.set_image(Gtk.Image.new_from_icon_name("", Gtk.IconSize.MENU))
                    uninstall_item.connect("activate", self.on_remove_button_clicked, [package_name])
                    menu.append(uninstall_item)

                    reinstall_item = Gtk.ImageMenuItem.new_with_label("♻️ Reinstall")
                    reinstall_item.set_image(Gtk.Image.new_from_icon_name("", Gtk.IconSize.MENU))
                    reinstall_item.connect("activate", self.on_reinstall_button_clicked, [package_name])
                    menu.append(reinstall_item)
                elif page_num == 2:
                    install_item = Gtk.ImageMenuItem.new_with_label("🟢️ Install")
                    install_item.set_image(Gtk.Image.new_from_icon_name("", Gtk.IconSize.MENU))
                    install_item.connect("activate", self.on_install_button_clicked, [package_name])
                    menu.append(install_item)

                menu.show_all()
                menu.popup_at_pointer(event)
            return True
        return False

    def _parse_repoquery_output(self, liststore, stdout, list_type=None):
        """Parses the output of 'dnf repoquery' and populates a ListStore."""
        if list_type is None:
            return
            
        for line in stdout.splitlines():
            line = line.strip()
            if not line:
                continue
            parts = line.split(':', 3)
            if len(parts) == 4:
                name = parts[0].strip()
                summary = parts[1].strip()
                repo = parts[2].strip()
                version_raw = parts[3].strip()
                
                version = re.sub(r'pclos20\d{2}\.x86_64|pclos20\d{2}\.i686|pclos20\d{2}', '', version_raw)
                version = version.rstrip('.-_')
                
                if list_type == "upgradable":
                    status = "Upgradable"
                    self.upgradable_packages_set.add(name)
                    liststore.append([False, name, version, summary, status])
                elif list_type == "installed":
                    if name not in self.upgradable_packages_set:
                        status = "Installed"
                        self.installed_packages_set.add(name)
                        liststore.append([False, name, version, summary, status])
                elif list_type == "available":
                    if name not in self.upgradable_packages_set and name not in self.installed_packages_set:
                        status = "Available"
                        liststore.append([False, name, version, summary, status])
                else:
                    status = "Unknown"

    def _parse_check_update_output(self, liststore, stdout):
        """Parses the output of 'dnf check-update' for upgradable packages."""
        upgradable_packages = {}
        for line in stdout.splitlines():
            line = line.strip()
            # Skip header lines from DNF
            if not line or "Last metadata expiration check:" in line or "Upgraded" in line or "Security" in line:
                continue
            
            # The output format is simple: package.arch version.release repo
            # We need to handle potential repo-less lines too.
            parts = line.split()
            if len(parts) >= 3:
                name_and_arch = parts[0].strip()
                version = parts[1].strip()
                repo = parts[2].strip()

                # Remove the architecture from the name for a cleaner look
                name_without_arch = name_and_arch.rsplit('.', 1)[0]
                upgradable_packages[name_without_arch] = {
                    "version": version,
                    "repo": repo,
                    "summary": "Upgradable package" # Placeholder
                }
        
        # Now, populate the liststore based on the parsed data
        for name, data in upgradable_packages.items():
            liststore.append([
                False,
                name,
                data["version"],
                data["summary"],
                "Upgradable"
            ])
            self.upgradable_packages_set.add(name)

    def _process_and_update_gui(self, upgradable_stdout, installed_stdout, available_stdout):
        """Parses the command output and updates the list stores."""
        self.upgradable_liststore.clear()
        self.installed_liststore.clear()
        self.available_packages_liststore.clear()
        self.upgradable_packages_set = set()
        self.installed_packages_set = set()
        
        # Step 1: Parse the check-update output first to get upgradable packages
        self._parse_check_update_output(self.upgradable_liststore, upgradable_stdout)

        # Step 2: Use repoquery to get installed packages, excluding those already identified as upgradable
        self._parse_repoquery_output(self.installed_liststore, installed_stdout, list_type="installed")

        # Step 3: Use repoquery to get available packages, excluding both upgradable and installed
        self._parse_repoquery_output(self.available_packages_liststore, available_stdout, list_type="available")

        self.cache_data()
        self._finish_full_refresh()

    def start_backend_refresh(self):
        """Starts a full refresh of all package lists in a background thread."""
        self.log("Starting full backend refresh...")
        self.start_spinner()

        def run_all_queries_and_process():
            try:
                self.log("Querying for upgradable packages...")
                upgradable_stdout, _, _ = self._run_query(
                    [self.dnf_cmd, "check-update", "--refresh"]
                )
                self.log("Querying for installed packages...")
                installed_stdout, _, _ = self._run_query(
                    [self.dnf_cmd, "repoquery", "--installed", "--queryformat", "%{name}:%{summary}:%{repoid}:%{evr}\n"]
                )
                self.log("Querying for available packages...")
                available_stdout, _, _ = self._run_query(
                    [self.dnf_cmd, "repoquery", "--available", "--queryformat", "%{name}:%{summary}:%{repoid}:%{evr}\n"]
                )
                GLib.idle_add(self._process_and_update_gui, upgradable_stdout, installed_stdout, available_stdout)
            except Exception as e:
                GLib.idle_add(self.log, f"An unexpected error during refresh: {e}")
                self.stop_spinner()

        threading.Thread(target=run_all_queries_and_process).start()
    
    def on_load_button_clicked(self, button=None):
        self.notebook.handler_block_by_func(self.on_notebook_page_changed)
        self.start_backend_refresh()

    def _finish_full_refresh(self):
        self.log("Full backend refresh completed.")
        self._update_gui_from_cache()
        self.notebook.handler_unblock_by_func(self.on_notebook_page_changed)
        self.stop_spinner()

    def on_pin_unpin_button_clicked(self, widget):
        self.PinDialog(self, self.dnf_cmd, self.log, self.run_command_in_thread)
    
    def on_update_button_clicked(self, button=None, packages=None):
        if packages is None:
            selected = self._get_selected_packages()
        else:
            selected = packages
        self._show_confirmation_dialog("upgrade", selected)

    def on_install_button_clicked(self, button=None, packages=None):
        if packages is None:
            selected = self._get_selected_packages()
        else:
            selected = packages
        self._show_confirmation_dialog("install", selected)

    def on_remove_button_clicked(self, button=None, packages=None):
        if packages is None:
            selected = self._get_selected_packages()
        else:
            selected = packages
        self._show_removal_confirmation_dialog(selected)

    def on_reinstall_button_clicked(self, button=None, packages=None):
        if packages is None:
            selected = self._get_selected_packages()
        else:
            selected = packages
        self._show_confirmation_dialog("reinstall", selected)

    def on_properties_button_clicked(self, button=None, packages=None):
        if packages is None:
            selected_packages = self._get_selected_packages()
        else:
            selected_packages = packages
        if len(selected_packages) != 1:
            self.log("Please select exactly one package to view its properties.")
            return
        package = selected_packages[0]
        self.log(f"Getting properties for {package}...")
        self.run_command_in_thread(
            [self.dnf_cmd, "info", package],
            f"Properties for {package} received.",
            f"Failed to get properties for {package}.",
            on_finish_callback=lambda stdout, stderr, returncode: self._show_info_dialog(package, "Properties", stdout),
            use_pkexec=False
        )

    def on_dependencies_button_clicked(self, button=None, packages=None):
        if packages is None:
            selected_packages = self._get_selected_packages()
        else:
            selected_packages = packages
        if len(selected_packages) != 1:
            self.log("Please select exactly one package to view its dependencies.")
            return
        package = selected_packages[0]
        self.log(f"Getting dependencies for {package}...")
        self.run_command_in_thread(
            [self.dnf_cmd, "repoquery", "--requires", package],
            f"Dependencies for {package} received.",
            f"Failed to get dependencies for {package}.",
            on_finish_callback=lambda stdout, stderr, returncode: self._show_info_dialog(package, "Dependencies", stdout),
            use_pkexec=False
        )

    def _show_info_dialog(self, package_name, info_type, info_text):
        if not info_text:
            self.log(f"No {info_type.lower()} found for {package_name}.")
            return
        InfoDialog(self, package_name, info_type, info_text)

    def on_row_activated(self, treeview, path, column):
        active_tab_index = self.notebook.get_current_page()
        if active_tab_index in [0, 1, 2]:
            model = treeview.get_model()
            treeiter = model.get_iter(path)
            if treeview:
                package_name = model.get_value(treeiter, 1)
                self.on_properties_button_clicked(None, packages=[package_name])

    def on_repositories_button_clicked(self, widget):
        self.log("Opening Repositories Dialog...")
        RepositoriesDialog(
            self,
            self.dnf_cmd,
            self.log,
            self.run_command_in_thread,
            self.start_backend_refresh
        )
    
    # --- New ProgressDialog Class ---
    class ProgressDialog(Gtk.Dialog):
        def __init__(self, parent, action, packages):
            super().__init__(
                title=f"Running {action.capitalize()}...",
                parent=parent,
                flags=0
            )
            self.action = action
            self.packages = packages
            self.set_default_size(600, 300)
            self.set_decorated(True)
            self.set_resizable(False)

            vbox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
            vbox.set_border_width(20)
            self.get_content_area().add(vbox)

            header_label = Gtk.Label(label=f"<b><big>Running DNF {action}</big></b>", use_markup=True)
            vbox.pack_start(header_label, False, False, 0)
            
            self.status_label = Gtk.Label(label="Initializing...", xalign=0)
            vbox.pack_start(self.status_label, False, False, 0)

            scrolled_window = Gtk.ScrolledWindow()
            scrolled_window.set_shadow_type(Gtk.ShadowType.IN)
            scrolled_window.set_size_request(-1, 125)
            vbox.pack_start(scrolled_window, True, True, 0)

            self.output_textview = Gtk.TextView()
            self.output_textview.set_editable(False)
            self.output_textview.set_cursor_visible(False)
            self.output_textview.set_wrap_mode(Gtk.WrapMode.WORD)
            scrolled_window.add(self.output_textview)

            self.show_all()

    def _show_confirmation_dialog(self, action, packages):
        if self.preferences.get("disable_confirm", False):
            self._execute_action(action, packages)
            return

        dialog = Gtk.MessageDialog(
            parent=self,
            flags=0,
            message_type=Gtk.MessageType.QUESTION,
            buttons=Gtk.ButtonsType.OK_CANCEL,
            text=f"Are you sure you want to {action} these packages?"
        )
        dialog.format_secondary_text("\n".join(packages))
        response = dialog.run()
        dialog.destroy()

        if response == Gtk.ResponseType.OK:
            self._execute_action(action, packages)
    
    def _execute_action(self, action, packages):
        if not packages:
            self.log("No packages selected.")
            return

        # Show the progress dialog first
        self.progress_dialog = self.ProgressDialog(self, action, packages)
        self.set_sensitive(False) # Disable the main window

        # Define the DNF command
        command_map = {
            "upgrade": [self.dnf_cmd, "distro-sync", "-y"],
            "install": [self.dnf_cmd, "install", "-y"],
            "remove": [self.dnf_cmd, "remove", "-y"],
            "reinstall": [self.dnf_cmd, "reinstall", "-y"]
        }

        if action not in command_map:
            self.log(f"Unknown action: {action}")
            return
        
        command = command_map[action] + packages

        # Run the command in a background thread
        thread = threading.Thread(
            target=self._run_transaction_in_thread,
            args=(command, )
        )
        thread.daemon = True
        thread.start()

    def _run_transaction_in_thread(self, command_parts):
        try:
            full_command = ["pkexec"] + command_parts
            GLib.idle_add(self.log, f"Executing command: {' '.join(full_command)}")
            GLib.idle_add(self.progress_dialog.status_label.set_text, "Starting transaction...")

            process = subprocess.Popen(
                full_command,
                stdout=subprocess.PIPE,
                stderr=subprocess.STDOUT,
                text=True,
                errors='ignore',
                env=os.environ
            )

            # Read stdout line-by-line in real-time
            for line in iter(process.stdout.readline, ''):
                GLib.idle_add(self._on_progress_update, line)
            
            # Wait for the process to finish and get the return code
            process.wait()
            stdout, stderr = process.communicate()
            returncode = process.returncode
            
            # Clean up the output to be displayed
            full_output = stdout.strip()
            if stderr.strip():
                full_output += "\n\nError Output:\n" + stderr.strip()

            GLib.idle_add(self.progress_dialog.output_textview.get_buffer().set_text, full_output)
            
            if returncode == 0:
                GLib.idle_add(self._on_transaction_complete, True)
            else:
                GLib.idle_add(self._on_transaction_complete, False)
                
        except FileNotFoundError:
            GLib.idle_add(self._on_transaction_complete, False, "pkexec or dnf not found. Is it installed?")
        except Exception as e:
            GLib.idle_add(self._on_transaction_complete, False, f"An unexpected error occurred: {e}")

    def _on_progress_update(self, line):
        # This function updates the progress dialog from the background thread
        buffer = self.progress_dialog.output_textview.get_buffer()
        buffer.insert(buffer.get_end_iter(), line)
        end_iter = buffer.get_end_iter()
        mark = buffer.create_mark(None, end_iter, False)
        self.progress_dialog.output_textview.scroll_to_mark(mark, 0.0, True, 0.0, 1.0)
        buffer.delete_mark(mark)

        if "Downloading packages" in line or "Downloading metadata" in line:
            self.progress_dialog.status_label.set_text("Downloading packages...")
        elif "Installing" in line:
            self.progress_dialog.status_label.set_text("Installing packages...")
        elif "Upgrading" in line:
            self.progress_dialog.status_label.set_text("Upgrading packages...")
        elif "Removing" in line:
            self.progress_dialog.status_label.set_text("Removing packages...")
        elif "Transaction" in line:
            self.progress_dialog.status_label.set_text("Transaction complete.")
        
    def _on_transaction_complete(self, success, error_message=""):
        # This function is called when the transaction thread is finished
        if success:
            self.progress_dialog.set_title("Success")
            self.progress_dialog.status_label.set_text("Operation completed successfully.")
            self.log("Transaction completed successfully.")
            if self.preferences.get("clear_cache", False):
                self._run_dnf_clean()
            self.on_load_button_clicked()
        else:
            self.progress_dialog.set_title("Error")
            self.progress_dialog.status_label.set_text("Operation failed.")
            self.log("Transaction failed.")
            if error_message:
                self.log(error_message)

        # Allow the user to close the dialog
        self.progress_dialog.add_button(Gtk.STOCK_CLOSE, Gtk.ResponseType.CLOSE)
        self.progress_dialog.connect("response", lambda d, r: d.destroy())
        self.set_sensitive(True)

    def _show_removal_confirmation_dialog(self, packages):
        if not packages:
            self.log("No packages selected for removal.")
            return

        self.log(f"Performing dry-run for removal of {', '.join(packages)}...")
        self.run_command_in_thread(
            [self.dnf_cmd, "remove", "--assumeno"] + packages,
            "", "",
            on_finish_callback=lambda stdout, stderr, returncode: self._on_removal_dry_run_complete(
                stdout, stderr, returncode, packages
            )
        )

    def _on_removal_dry_run_complete(self, stdout, stderr, returncode, selected_packages):
        all_packages_to_remove = []
        parsing = False
        
        for line in stdout.splitlines():
            line = line.strip()
            
            if line.startswith("Removing:") or line.startswith("Removing dependent packages:"):
                parsing = True
                continue
            
            if line.startswith("Transaction Summary"):
                break
            
            if parsing and line:
                parts = line.split()
                if parts:
                    all_packages_to_remove.append(parts[0])

        if not all_packages_to_remove:
            error_msg = f"Failed to perform dry-run for removal. DNF returned an error.\n\n" \
                        f"This may be because one or more packages are essential system components " \
                        f"and cannot be removed, or a problem occurred with DNF itself.\n\n" \
                        f"Error Details: {stderr.strip()}"
            
            error_dialog = Gtk.MessageDialog(
                parent=self,
                flags=0,
                message_type=Gtk.MessageType.ERROR,
                buttons=Gtk.ButtonsType.OK,
                text="Removal Failed"
            )
            error_dialog.format_secondary_text(error_msg)
            error_dialog.run()
            error_dialog.destroy()
            self.log(f"Dry-run for removal failed. Error: {stderr.strip()}")
            return
        
        dialog = Gtk.Dialog(
            title="Confirm Package Removal",
            parent=self,
            flags=0,
            buttons=(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
                     Gtk.STOCK_OK, Gtk.ResponseType.OK)
        )
        
        content_area = dialog.get_content_area()
        
        main_label = Gtk.Label(
            label="The following packages will be removed:\n",
            xalign=0
        )
        main_label.set_margin_top(12)
        main_label.set_margin_start(12)
        main_label.set_margin_end(12)
        content_area.pack_start(main_label, False, False, 0)
        
        package_list_text = ""
        for pkg in all_packages_to_remove:
            package_list_text += f"• <b>{pkg}</b>\n"
        
        scrolled_window = Gtk.ScrolledWindow()
        scrolled_window.set_size_request(400, 200)
        scrolled_window.set_policy(Gtk.PolicyType.AUTOMATIC, Gtk.PolicyType.AUTOMATIC)
        scrolled_window.set_margin_bottom(12)
        scrolled_window.set_margin_start(12)
        scrolled_window.set_margin_end(12)
        
        list_label = Gtk.Label(xalign=0, yalign=0)
        list_label.set_markup(package_list_text)
        list_label.set_selectable(True)
        list_label.set_line_wrap(False)
        
        scrolled_window.add(list_label)
        
        content_area.pack_start(scrolled_window, True, True, 0)
        
        dialog.show_all()
        response = dialog.run()
        dialog.destroy()
        
        if response == Gtk.ResponseType.OK:
            GLib.idle_add(self._execute_action, "remove", all_packages_to_remove)
        else:
            self.log("Removal operation cancelled by user.")
    
    def _update_gui_from_cache(self):
        self.log("Updating GUI lists from cache...")
        self.upgradable_filter.refilter()
        self.installed_filter.refilter()
        self.available_packages_filter.refilter()
        self.update_statusbar_count()
        self._update_button_sensitivity()
        self.log(f"Populated {len(self.upgradable_liststore)} upgradable packages.")
        self.log(f"Populated {len(self.installed_liststore)} installed packages.")
        self.log(f"Populated {len(self.available_packages_liststore)} available packages.")
    
    def run_command_in_thread(self, command_parts, success_message, error_message, on_finish_callback=None, use_pkexec=False):
        self.start_spinner()
        def run_task():
            try:
                full_command = ["pkexec"] + command_parts if use_pkexec else command_parts
                self.log(f"Executing command: {' '.join(full_command)}")
                process = subprocess.Popen(
                    full_command,
                    stdout=subprocess.PIPE,
                    stderr=subprocess.PIPE,
                    text=True,
                    errors='ignore',
                    env=os.environ
                )
                stdout, stderr = process.communicate()
                returncode = process.returncode
            except FileNotFoundError:
                GLib.idle_add(self.log, "Error: The required command was not found. Please ensure dnf and pkexec are installed and in your PATH.")
                returncode = 1
                stdout, stderr = "", "Command not found"
            except Exception as e:
                GLib.idle_add(self.log, f"An unexpected error occurred: {e}")
                returncode = 1
                stdout, stderr = "", str(e)
            finally:
                if returncode == 0:
                    GLib.idle_add(self.log, success_message)
                    # Log the standard output only on success
                    if stdout:
                        GLib.idle_add(self.log, stdout.strip())
                else:
                    # Log the error message and stderr on failure
                    GLib.idle_add(self.log, f"Error: {error_message}")
                    if stderr:
                        GLib.idle_add(self.log, stderr.strip())
                    
                if on_finish_callback:
                    GLib.idle_add(on_finish_callback, stdout, stderr, returncode)
                
                GLib.idle_add(self.stop_spinner)
        threading.Thread(target=run_task).start()
    def _run_query(self, command_parts):
        """
        Synchronously runs a dnf query and returns the output.
        Designed for a dedicated worker thread.
        """
        try:
            process = subprocess.Popen(
                command_parts,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True,
                errors='ignore',
                env=os.environ
            )
            stdout, stderr = process.communicate()
            return stdout, stderr, process.returncode
        except FileNotFoundError:
            self.log(f"Error: {command_parts[0]} not found. Is it installed?")
            return "", "", 1
        except Exception as e:
            self.log(f"An unexpected error occurred during command execution: {e}")
            return "", "", 1

    def _run_dnf_clean(self):
        """Runs `dnf clean all` with pkexec if enabled."""
        self.log(f"Cleaning {self.dnf_cmd} cache...")
        command = ["pkexec", self.dnf_cmd, "clean", "all"]
        try:
            subprocess.run(command, check=True, text=True, errors='ignore', capture_output=True)
            self.log("DNF cache cleared.")
        except (subprocess.CalledProcessError, FileNotFoundError) as e:
            self.log(f"Error clearing cache: {e}")

    def load_data_from_cache(self):
        """Loads package data from the cache file."""
        try:
            with open(os.path.join(CONFIG_DIR, "cache.json"), 'r') as f:
                data = json.load(f)
            self.log("Successfully read data from cache.")
            self.upgradable_packages_set = {row[1] for row in data.get("upgradable", [])}
            self.installed_packages_set = {row[1] for row in data.get("installed", [])}
            
            self._populate_liststore(self.upgradable_liststore, data.get("upgradable", []))
            self._populate_liststore(self.installed_liststore, data.get("installed", []))
            self._populate_liststore(self.available_packages_liststore, data.get("available", []))
            return True
        except (IOError, json.JSONDecodeError):
            self.log("Failed to read from cache.")
            return False

    def _populate_liststore(self, liststore, data):
        liststore.clear()
        for row in data:
            if len(row) == 5:
                liststore.append(row)

    def cache_data(self):
        """Saves package data to the cache file."""
        data = {
            "upgradable": [list(row) for row in self.upgradable_liststore],
            "installed": [list(row) for row in self.installed_liststore],
            "available": [list(row) for row in self.available_packages_liststore],
        }
        try:
            if not os.path.exists(CONFIG_DIR):
                os.makedirs(CONFIG_DIR)
            with open(os.path.join(CONFIG_DIR, "cache.json"), 'w') as f:
                json.dump(data, f, indent=4)
            self.log("Data successfully written to cache.")
        except (IOError, json.JSONDecodeError) as e:
            self.log(f"Error writing to cache: {e}")

    def _get_current_desktop_environment(self):
        xdg_desktop = os.environ.get("XDG_CURRENT_DESKTOP", "").lower()
        desktop_session = os.environ.get("DESKTOP_SESSION", "").lower()
        if "kde" in xdg_desktop or "plasma" in xdg_desktop or "kde" in desktop_session:
            return "kde"
        if "gnome" in xdg_desktop or "gnome" in xdg_desktop or "gnome" in desktop_session:
            return "gnome"
        if "mate" in xdg_desktop or "mate" in desktop_session:
            return "mate"
        if "xfce" in xdg_desktop or "xfce" in desktop_session:
            return "xfce"
        return "unknown"

    def apply_custom_css(self):
        current_de = self._get_current_desktop_environment()
        self.log(f"Detected desktop environment: {current_de.upper()}.")
        if current_de == "kde":
            self.log("Applying custom CSS for button borders.")
            self.style_provider = Gtk.CssProvider()
            self.style_provider.load_from_data(CUSTOM_CSS)
            Gtk.StyleContext.add_provider_for_screen(
                Gdk.Screen.get_default(),
                self.style_provider,
                Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION
            )
        else:
            self.log("Not applying custom CSS. Relying on system theme.")

    def load_preferences(self):
        if not os.path.exists(CONFIG_DIR):
            os.makedirs(CONFIG_DIR)
        preferences = {
            "history_size": 50,
            "disable_confirm": False,
            "clear_cache": False,
            "keepcache": False,
            "dnf_cmd": _detect_dnf_command(),
            "window_width": 1000,
            "window_height": 700,
            "paned_position": 0,
            "show_pin_dialog": False
        }
        if os.path.exists(CONFIG_FILE):
            try:
                with open(CONFIG_FILE, 'r') as f:
                    saved_prefs = json.load(f)
                    preferences.update(saved_prefs)
            except (IOError, json.JSONDecodeError) as e:
                self.log(f"Warning: Failed to load preferences from {CONFIG_FILE}. Using defaults. Error: {e}")
        return preferences

    def save_preferences(self, preferences):
        preferences["window_width"], preferences["window_height"] = self.get_size()
        preferences["paned_position"] = self.main_paned.get_position()

        if self.dnf_cmd != preferences.get("dnf_cmd", DNF_CMD):
            self.dnf_cmd = preferences.get("dnf_cmd", DNF_CMD)
            self.log(f"Package manager command changed to: {self.dnf_cmd}")
        try:
            with open(CONFIG_FILE, 'w') as f:
                json.dump(preferences, f, indent=4)
            self.log("Preferences saved.")
        except IOError as e:
            self.log(f"Error: Failed to save preferences to {CONFIG_FILE}. Error: {e}")

    def on_preferences_button_clicked(self, widget, data=None):
        self.PreferencesWindow(self, self.preferences)
        
    def on_history_button_clicked(self, widget, data=None):
        self.HistoryDialog(self)

    def on_about_button_clicked(self, widget):
        about_dialog = Gtk.AboutDialog()
        about_dialog.set_program_name("DNF Package Manager")
        about_dialog.set_version("1.1")
        about_dialog.set_copyright("https://www.pclinuxos.com/\n\n© 2025 PCLinuxOS Community")
        about_dialog.set_comments("A simple and user-friendly GUI for DNF to manage packages.")
        about_dialog.set_authors(["Upgreyed, Texstar"])
        
        logo_path = os.path.join(os.path.dirname(os.path.abspath(__file__)), "/usr/share/pixmaps/dnf-package-manager.png")
        if os.path.exists(logo_path):
            try:
                pixbuf = GdkPixbuf.Pixbuf.new_from_file_at_size(logo_path, 64, 64)
                about_dialog.set_logo(pixbuf)
            except GLib.Error as e:
                self.log(f"Error loading logo: {e}")
        else:
            self.log(f"Logo file not found: {logo_path}")

        about_dialog.run()
        about_dialog.destroy()

    def start_spinner(self, message=""):
        self.spinner.set_tooltip_text(f"🕓️ Operation in progress: {message}")
        self.spinner.start()
        self.spinner.show()

    def stop_spinner(self):
        self.spinner.stop()
        self.spinner.hide()

    def on_log_toggled(self, button):
        if button.get_active():
            self.log_arrow.set_property("arrow_type", Gtk.ArrowType.UP)

            main_x, main_y = self.get_position()
            main_width, main_height = self.get_size()

            log_width = main_width
            log_height = 150
            self.log_window.set_size_request(log_width, log_height)
            self.log_window.move(main_x, main_y + main_height - log_height)
            self.log_window.show_all()
        else:
            self.log_arrow.set_property("arrow_type", Gtk.ArrowType.DOWN)
            self.log_window.hide()

    def set_default_button_states(self):
        self.update_button.set_sensitive(False)
        self.install_button.set_sensitive(False)
        self.remove_button.set_sensitive(False)
        self.reinstall_button.set_sensitive(False)
        self.description_button.set_sensitive(False)

    def _update_button_sensitivity(self):
        tab_index = self.notebook.get_current_page()
        self.set_default_button_states()

        selected_count = len(self._get_selected_packages())
        is_single_selection = (selected_count == 1)
        is_any_selection = (selected_count > 0)

        if tab_index == 0:
            self.update_button.set_sensitive(is_any_selection)
            self.description_button.set_sensitive(is_single_selection)
        elif tab_index == 1:
            self.remove_button.set_sensitive(is_any_selection)
            self.reinstall_button.set_sensitive(is_any_selection)
            self.description_button.set_sensitive(is_single_selection)
        elif tab_index == 2:
            self.install_button.set_sensitive(is_any_selection)
            self.description_button.set_sensitive(is_single_selection)

    def on_notebook_page_changed(self, notebook, page, page_num):
        self.search_entry.set_text("")
        self.do_search(page_num)
        self._update_button_sensitivity()

    def on_search_changed(self, entry=None):
        GLib.idle_add(self.do_search)

    def on_select_all_button_clicked(self, button):
        tab_index = self.notebook.get_current_page()

        current_filter = None
        if tab_index == 0:
            current_filter = self.upgradable_filter
        elif tab_index == 1:
            current_filter = self.installed_filter
        elif tab_index == 2:
            current_filter = self.available_packages_filter

        if not current_filter: return

        all_selected = all(row[0] for row in current_filter)
        is_active = not all_selected

        treeiter = current_filter.get_iter_first()
        while treeiter:
            child_path = current_filter.convert_path_to_child_path(current_filter.get_path(treeiter))
            if child_path:
                child_iter = current_filter.get_model().get_iter(child_path)
                current_filter.get_model()[child_iter][0] = is_active
            treeiter = current_filter.iter_next(treeiter)
        self._update_button_sensitivity()

    def on_toggled_package(self, cell, path, liststore_filter):
        treepath = Gtk.TreePath(path)
        child_path = liststore_filter.convert_path_to_child_path(treepath)
        if child_path:
            liststore = liststore_filter.get_model()
            child_iter = liststore.get_iter(child_path)
            if child_iter:
                liststore[child_iter][0] = not liststore[child_iter][0]
                self._update_button_sensitivity()

    def log(self, message):
        timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        formatted_message = f"[{timestamp}] {message}"
        print(f"[GUI Log] {formatted_message}")
        def _add_log_message():
            buffer = self.log_textview.get_buffer()
            buffer.insert(buffer.get_end_iter(), formatted_message + "\n")
            end_iter = buffer.get_end_iter()
            mark = buffer.create_mark(None, end_iter, False)
            self.log_textview.scroll_to_mark(mark, 0.0, True, 0.0, 1.0)
            buffer.delete_mark(mark)
        GLib.idle_add(_add_log_message)

    def filter_packages(self, model, iter, data):
        search_text = self.search_entry.get_text().lower()
        if not search_text:
            return True
        package_name = model.get_value(iter, 1).lower()
        package_summary = model.get_value(iter, 3).lower()
        return search_text in package_name or search_text in package_summary

    def update_statusbar_count(self, page_num=None):
        search_text = self.search_entry.get_text().lower()
        search_active = " (Search active)" if search_text else ""
        tab_index = page_num if page_num is not None else self.notebook.get_current_page()
        if tab_index in self.status_bar_map:
            label_text, filter_obj, liststore_obj = self.status_bar_map[tab_index]
            filtered_count = filter_obj.iter_n_children(None)
            total_count = len(liststore_obj)
            self.statusbar_label.set_text(f"{label_text}: {filtered_count} of {total_count}{search_active}")
        else:
            self.statusbar_label.set_text("Ready.")

    def do_search(self, page_num=None):
        self.upgradable_filter.refilter()
        self.installed_filter.refilter()
        self.available_packages_filter.refilter()
        self.update_statusbar_count(page_num)
        return False

    def _get_selected_packages(self):
        active_tab_index = self.notebook.get_current_page()
        model = None
        if active_tab_index == 0:
            model = self.upgradable_liststore
        elif active_tab_index == 1:
            model = self.installed_liststore
        elif active_tab_index == 2:
            model = self.available_packages_liststore

        selected_packages = []
        if model:
            for row in model:
                if row[0]:
                    selected_packages.append(row[1])
        return selected_packages

# --- Application Entry Point ---

def on_activate(app):
    win = PackageApp(app)
    win.show_all()

def main():
    app = Gtk.Application.new("com.example.dnf5-frontend", Gio.ApplicationFlags.FLAGS_NONE)
    app.connect("activate", on_activate)
    app.run(sys.argv)

if __name__ == '__main__':
    main()
