"""
VideoCard potenziata con supporto alla coda, integrazione VLC e gestione video completa.
"""

# Enhanced VideoCard with queue support, VLC integration, and comprehensive video management.
from PyQt5.QtWidgets import (
    QFrame, QLabel, QVBoxLayout, QHBoxLayout, QMenu, QPushButton, QCheckBox,
    QMessageBox, QFileDialog, QProgressDialog, QInputDialog, QApplication
)
from PyQt5.QtCore import Qt, pyqtSignal, QTimer, QMetaObject, Q_ARG, QSize, QPoint, QThreadPool, QObject
from PyQt5.QtGui import QPixmap, QIcon, QMovie
import sys
import os
import subprocess
import time
import threading
import psutil
from pathlib import Path
from typing import Optional, List
import webbrowser
import requests
import yt_dlp
import logging
from bs4 import BeautifulSoup
from urllib.parse import urlparse
import re

from config.logging_config import logger
from config.settings import (
    PlayerConfig, DOWNLOAD_DIR, APP_NAME
)
from config.theme import Theme
from core.models import Video, Playlist
from core.services.playlist_service import PlaylistService
from core.services.rss_service import RSSService
from core.player import DownloadWorker, DirectURLWorker

# Ottieni le costanti da PlayerConfig
# Get constants from PlayerConfig
DEFAULT_VOLUME = PlayerConfig.DEFAULT_VOLUME
DEFAULT_PLAYBACK_SPEED = PlayerConfig.DEFAULT_PLAYBACK_SPEED

class DownloadSignals(QObject):
    progress = pyqtSignal(int)
    finished = pyqtSignal(str, str)
    error = pyqtSignal(str, str)

class LoadingSpinner(QFrame):
    """Widget spinner di caricamento per operazioni asincrone."""
    # Loading spinner widget for async operations.
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setFixedSize(60, 60)
        self.setAttribute(Qt.WA_TranslucentBackground)
        self.setWindowFlags(Qt.FramelessWindowHint)
        self.setFocusPolicy(Qt.NoFocus)
        self.setStyleSheet(f"""
            LoadingSpinner {{
                background: transparent;
                border: none;
            }}
            QLabel {{
                background: transparent;
                border: none;
            }}
        """)
        layout = QVBoxLayout(self)
        layout.setContentsMargins(0, 0, 0, 0)
        layout.setSpacing(0)
        
        self.spinner_label = QLabel()
        self.spinner_label.setFixedSize(60, 60)
        self.spinner_label.setAlignment(Qt.AlignCenter)
        self.spinner_label.setStyleSheet(f"""
            QLabel {{
                background: transparent;
                border: none;
            }}
        """)
        
        self.movie = QMovie("spinner.gif")
        self.movie.setScaledSize(QSize(60, 60))
        self.spinner_label.setMovie(self.movie)
        layout.addWidget(self.spinner_label)

    def showEvent(self, event):
        super().showEvent(event)
        self.movie.start()

    def hideEvent(self, event):
        super().hideEvent(event)
        self.movie.stop()

class VideoCard(QFrame):
    """
    Widget VideoCard con funzionalità complete per:
    - Riproduzione video (VLC)
    - Gestione coda
    - Integrazione playlist
    - Gestione feed RSS
    - Supporto download
    """
    # Video card widget with full functionality for:
    # - Video playback (VLC)
    # - Queue management
    # - Playlist integration
    # - RSS feed management
    # - Download support
    
    watch_later_removed = pyqtSignal()
    video_removed = pyqtSignal(str)
    play_requested = pyqtSignal(Video)          # Segnale per la riproduzione di un video
    # Signal for playing a video
    add_to_queue_requested = pyqtSignal(Video)   # Segnale per aggiungere alla coda
    # Signal for adding to queue

    def __init__(self, video: Video, playlist_id: int = None, main_window=None, show_selector: bool = False):
        super().__init__()
        self.video = video
        self.playlist_id = playlist_id
        self.main_window = main_window
        self.show_selector = show_selector
        self.thumbnail_loader = None
        
        self.setFixedSize(530, 340)
        self.setStyleSheet(Theme.video_card_style())
        self._setup_ui()
        
        self.loading_spinner = None
        self.setFocusPolicy(Qt.StrongFocus)
        self.playlist_service = PlaylistService()
        self.thread_pool = QThreadPool.globalInstance()

    def _setup_ui(self):
        """Configura l'interfaccia utente completa con tutte le funzionalità."""
        # Setup the complete UI with all functionality.
        layout = QVBoxLayout(self)
        layout.setContentsMargins(0, 0, 0, 0)
        layout.setSpacing(0)

        # Contenitore della miniatura
        # Thumbnail container
        self.thumb_container = QFrame()
        self.thumb_container.setFixedSize(530, 290)
        self.thumb_container.setStyleSheet(f"""
            QFrame {{
                background: {Theme.BACKGROUND};
                border-top-left-radius: 12px;
                border-top-right-radius: 12px;
                border: none;
            }}
        """)
        thumb_layout = QVBoxLayout(self.thumb_container)
        thumb_layout.setContentsMargins(0, 0, 0, 0)
        
        self.thumb_label = QLabel(self.thumb_container)
        self.thumb_label.setObjectName("thumb_label")
        self.thumb_label.setFixedSize(530, 280)
        self.thumb_label.setAlignment(Qt.AlignCenter)
        self.thumb_label.setStyleSheet(f"""
            QLabel#thumb_label {{
                background: {Theme.BACKGROUND};
                border-top-left-radius: 12px;
                border-top-right-radius: 12px;
                border: none;
            }}
        """)
        self.thumb_label.setScaledContents(True)
        thumb_layout.addWidget(self.thumb_label)

        # Pulsante di riproduzione sovrapposto
        # Play overlay button
        self.play_overlay = QPushButton("▶", self.thumb_container)
        self.play_overlay.setFixedSize(68, 48)
        self.play_overlay.setStyleSheet(f"""
            QPushButton {{
                background-color: rgba(255, 255, 255, 0.9);
                color: black;
                font-size: 20px;
                border-radius: 4px;
                border: none;
            }}
            QPushButton:hover {{
                background-color: white;
            }}
        """)
        self.play_overlay.hide()

        # Frame sovrapposto
        # Overlay frame
        self.overlay = QFrame(self.thumb_container)
        self.overlay.setFixedSize(560, 290)
        self.overlay.setStyleSheet(f"""
            QFrame {{
                background-color: rgba(0, 0, 0, 0.3);
                border-radius: 12px;
            }}
        """)
        self.overlay.hide()

        # Etichetta della durata
        # Duration label
        self.duration_label = QLabel(self.thumb_container)
        self.duration_label.setObjectName("duration")
        self.duration_label.setText(f" {self._format_duration(self.video.duration)} ")
        self.duration_label.setStyleSheet(f"""
            QLabel#duration {{
                background-color: rgba(0, 0, 0, 0.8);
                color: white;
                border: none;
                border-radius: 2px;
                padding: 2px 4px;
                font-size: 12px;
                font-weight: 500;
                min-width: 40px;
                text-align: center;
                position: absolute;
                bottom: 8px;
                right: 8px;
            }}
        """)
        self.duration_label.move(470, 235)

        # Frame delle informazioni
        # Info frame
        info_frame = QFrame()
        info_frame.setFixedSize(530, 78)
        info_frame.setStyleSheet(f"""
            QFrame {{
                background: {Theme.SURFACE};
                border: none;
                border-bottom-left-radius: 12px;
                border-bottom-right-radius: 12px;
            }}
        """)
        info_layout = QVBoxLayout(info_frame)
        info_layout.setContentsMargins(12, 8, 12, 8)
        info_layout.setSpacing(4)

        # Titolo
        # Title
        title_text = self.video.title[:50] + "..." if len(self.video.title) > 50 else self.video.title
        self.title_label = QLabel(title_text)
        self.title_label.setObjectName("title")
        self.title_label.setWordWrap(True)
        self.title_label.setToolTip(self.video.title)
        self.title_label.setStyleSheet(f"""
            QLabel#title {{
                color: {Theme.TEXT_PRIMARY};
                font-size: 14px;
                font-weight: 500;
            }}
        """)
        info_layout.addWidget(self.title_label)

        # Autore
        # Author
        if hasattr(self.video, 'author') and self.video.author:
            self.author_label = QLabel(f"👤 {self.video.author}")
            self.author_label.setObjectName("channel")
            self.author_label.setStyleSheet(f"""
                QLabel#channel {{
                    color: {Theme.TEXT_SECONDARY};
                    font-size: 12px;
                }}
            """)
            info_layout.addWidget(self.author_label)

        # Data di pubblicazione
        # Published date
        if hasattr(self.video, 'published') and self.video.published:
            pub_date = self.video.published.strftime("%d/%m/%Y")
            self.date_label = QLabel(f"📅 {pub_date}")
            self.date_label.setObjectName("date_label")
            self.date_label.setStyleSheet(f"""
                QLabel#date_label {{
                    color: {Theme.TEXT_SECONDARY};
                    font-size: 12px;
                }}
            """)
            info_layout.addWidget(self.date_label)

        # Checkbox per la selezione
        # Checkbox for selection
        if self.show_selector:
            self.checkbox = QCheckBox()
            info_layout.addWidget(self.checkbox)

        info_layout.addStretch()

        # Aggiungi i contenitori al layout principale
        # Add containers to main layout
        layout.addWidget(self.thumb_container)
        layout.addWidget(info_frame)

        # Gestione degli eventi
        # Event handling
        self.setMouseTracking(True)
        self.thumb_container.setMouseTracking(True)
        self.thumb_container.installEventFilter(self)
        self.setContextMenuPolicy(Qt.CustomContextMenu)
        self.customContextMenuRequested.connect(self.show_menu)

        # Carica la miniatura
        # Load thumbnail
        if hasattr(self.video, 'thumbnail_url') and self.video.thumbnail_url:
            self.load_thumbnail(self.video.thumbnail_url)

    def load_thumbnail(self, url):
        """Carica la miniatura in modo asincrono."""
        # Load thumbnail asynchronously.
        def _load():
            try:
                logger.info(f"Caricamento miniatura da: {url}")
                # Loading thumbnail from: {url}
                response = requests.get(url, timeout=10)
                response.raise_for_status()
                
                pixmap = QPixmap()
                if pixmap.loadFromData(response.content):
                    logger.info("Miniatura caricata con successo")
                    # Thumbnail loaded successfully
                    QMetaObject.invokeMethod(
                        self.thumb_label, "setPixmap",
                        Qt.QueuedConnection,
                        Q_ARG(QPixmap, pixmap.scaled(
                            self.thumb_label.size(),
                            Qt.KeepAspectRatio,
                            Qt.SmoothTransformation
                        ))
                    )
                else:
                    logger.error("Impossibile caricare la pixmap dai dati")
                    # Failed to load pixmap from data
                    QMetaObject.invokeMethod(
                        self.thumb_label, "setText",
                        Qt.QueuedConnection,
                        Q_ARG(str, "Impossibile caricare la miniatura")
                        # Thumbnail failed to load
                    )
            except Exception as e:
                logger.error(f"Errore nel caricamento della miniatura: {e}")
                # Error loading thumbnail: {e}
                QMetaObject.invokeMethod(
                    self.thumb_label, "setText",
                    Qt.QueuedConnection,
                    Q_ARG(str, "Impossibile caricare la miniatura")
                    # Thumbnail failed to load
                )

        threading.Thread(target=_load, daemon=True).start()

    def eventFilter(self, obj, event):
        """Gestisci gli eventi di hover per il contenitore della miniatura."""
        # Handle hover events for the thumbnail container.
        if obj is self.thumb_container:
            if event.type() == event.Enter:
                self.on_hover_enter()
            elif event.type() == event.Leave:
                self.on_hover_leave()
        return super().eventFilter(obj, event)

    def on_hover_enter(self):
        """Mostra sovrapposizione e pulsante di riproduzione al passaggio del mouse."""
        # Show overlay and play button on hover.
        self.overlay.show()
        self.play_overlay.show()
        self.play_overlay.move(
            (self.thumb_container.width() - self.play_overlay.width()) // 2,
            (self.thumb_container.height() - self.play_overlay.height()) // 2
        )
        self.play_overlay.raise_()
        self.overlay.raise_()

    def on_hover_leave(self):
        """Nasconde sovrapposizione e pulsante di riproduzione quando il mouse esce."""
        # Hide overlay and play button when hover ends.
        self.overlay.hide()
        self.play_overlay.hide()

    def _format_duration(self, seconds: str) -> str:
        """Formatta la stringa della durata in MM:SS o H:MM:SS."""
        # Format duration string to MM:SS or H:MM:SS.
        try:
            sec = int(seconds)
            h, m, s = sec // 3600, (sec % 3600) // 60, sec % 60
            if h:
                return f"{h}:{m:02d}:{s:02d}"
            return f"{m}:{s:02d}"
        except ValueError:
            return "0:00"

    def _center_spinner(self):
        """Centra lo spinner di caricamento sulla miniatura."""
        # Center the loading spinner on the thumbnail.
        if self.loading_spinner:
            self.loading_spinner.move(
                (self.thumb_label.width() - 60) // 2,
                (self.thumb_label.height() - 60) // 2
            )

    def _hide_loading_spinner(self):
        """Nasconde lo spinner di caricamento con un ritardo."""
        # Hide the loading spinner with a delay.
        QTimer.singleShot(0, self._safe_hide_spinner)

    def _safe_hide_spinner(self):
        """Nasconde e elimina in modo sicuro lo spinner di caricamento."""
        # Safely hide and delete the loading spinner.
        if self.loading_spinner:
            try:
                self.loading_spinner.hide()
                self.loading_spinner.deleteLater()
            except Exception:
                pass
            self.loading_spinner = None
        self.setEnabled(True)
        self.duration_label.raise_()

    def _is_process_active(self, pid):
        """Controlla se un processo è attivo."""
        # Check if a process is active.
        try:
            process = psutil.Process(pid)
            cpu_percent = process.cpu_percent(interval=0.1)
            return cpu_percent > 0
        except (psutil.NoSuchProcess, psutil.AccessDenied):
            return False

    def get_direct_url(self, video_url: str) -> str:
        """Ottieni l'URL diretto per un video usando yt-dlp."""
        # Get the direct URL for a video using yt-dlp.
        ydl_opts = {
            'quiet': True,
            'no_warnings': True,
            'force_generic_extractor': True,
            'format': 'best[height<=1080]',
            'get_url': True,
        }
        try:
            with yt_dlp.YoutubeDL(ydl_opts) as ydl:
                result = ydl.extract_info(video_url, download=False)
                if 'url' in result:
                    logger.info(f"URL diretto estratto con successo per: {video_url}")
                    # Successfully extracted direct URL for: {video_url}
                    return result['url']
                else:
                    logger.warning(f"Nessun URL diretto trovato per: {video_url}, utilizzo dell'URL originale")
                    # No direct URL found for: {video_url}, using original URL
                    return video_url
        except Exception as e:
            logger.error(f"Errore nell'ottenere l'URL diretto per {video_url}: {e}")
            # Error getting direct URL for {video_url}: {e}
            return video_url

    def play_with_vlc(self):
        """Riproduci video con VLC usando DirectURLWorker."""
        # Play video with VLC player using DirectURLWorker.
        self.play_requested.emit(self.video)
        if self.main_window and hasattr(self.main_window, 'show_vlc_player'):
            direct_url_worker = DirectURLWorker(self.video.url)
            
            def on_direct_url_ready(direct_url: str):
                logger.info(f"Riproduzione con VLC: {direct_url}")
                # Playing with VLC: {direct_url}
                self.main_window.show_vlc_player(self.video)
                
            def on_direct_url_error(error_msg: str):
                logger.error(f"Errore nella risoluzione dell'URL diretto: {error_msg}")
                # Error resolving direct URL: {error_msg}
                if self.main_window and hasattr(self.main_window, 'statusBar'):
                    self.main_window.statusBar().showMessage(f"Errore: {error_msg}")
                    # Error: {error_msg}
                    
            direct_url_worker.signals.finished.connect(on_direct_url_ready)
            direct_url_worker.signals.error.connect(on_direct_url_error)
            QThreadPool.globalInstance().start(direct_url_worker)

    def play_with_vlc_forced(self):
        """Force playback with VLC, ignoring user settings."""
        logger.info(f"🎬 [DEBUG] Forced VLC playback: {self.video.title}")
        try:
            if self.main_window:
                # Try multiple possible method names and window structures
                if hasattr(self.main_window, '_play_with_vlc_direct'):
                    self.main_window._play_with_vlc_direct(self.video)
                elif hasattr(self.main_window, 'play_with_vlc_direct'):
                    self.main_window.play_with_vlc_direct(self.video)
                elif hasattr(self.main_window, 'show_vlc_player'):
                    self.main_window.show_vlc_player(self.video)
                elif hasattr(self.main_window, 'play_video'):
                    self.main_window.play_video(self.video)
                else:
                    # If main_window is a QStackedWidget, try to find the actual main window
                    parent = self.main_window.parent()
                    while parent and not hasattr(parent, '_play_with_vlc_direct'):
                        parent = parent.parent()
                    
                    if parent and hasattr(parent, '_play_with_vlc_direct'):
                        parent._play_with_vlc_direct(self.video)
                    else:
                        # Fallback: use DirectURLWorker directly
                        self._fallback_vlc_playback()
            else:
                self._fallback_vlc_playback()
        except Exception as e:
            logger.error(f"Error in forced VLC playback: {e}")
            self._fallback_vlc_playback()

    def _fallback_vlc_playback(self):
        """Fallback VLC playback when main window methods are not available."""
        try:
            direct_url_worker = DirectURLWorker(self.video.url)
            
            def on_direct_url_ready(direct_url: str):
                logger.info(f"Fallback VLC playback with direct URL: {direct_url}")
                try:
                    # Try to play with VLC using subprocess
                    if sys.platform == "win32":
                        subprocess.Popen(['vlc', direct_url], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
                    else:
                        subprocess.Popen(['vlc', direct_url], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
                except Exception as e:
                    logger.error(f"Failed to start VLC: {e}")
                    # Final fallback: open in browser
                    webbrowser.open(self.video.url)
                    
            def on_direct_url_error(error_msg: str):
                logger.error(f"Fallback VLC playback failed: {error_msg}")
                webbrowser.open(self.video.url)
                    
            direct_url_worker.signals.finished.connect(on_direct_url_ready)
            direct_url_worker.signals.error.connect(on_direct_url_error)
            QThreadPool.globalInstance().start(direct_url_worker)
        except Exception as e:
            logger.error(f"Complete VLC playback failure: {e}")
            webbrowser.open(self.video.url)

    def mousePressEvent(self, event):
        """Gestisci gli eventi di pressione del mouse per la riproduzione video."""
        # Handle mouse press events for video playback.
        if event.button() == Qt.LeftButton and (not hasattr(self, 'checkbox') or not self.checkbox.underMouse()):
            # Left click: use the user's preference from settings
            self.play_requested.emit(self.video)
            if self.main_window and hasattr(self.main_window, 'play_video'):
                self.main_window.play_video(self.video)  # This will respect the user's setting
        super().mousePressEvent(event)

    def show_menu(self, pos):
        """Mostra il menu contestuale con tutte le opzioni video."""
        # Show context menu with all video options.
        menu = QMenu(self)
        menu.setStyleSheet(f"""
            QMenu {{
                background: {Theme.SURFACE};
                border: 1px solid {Theme.BORDER};
                border-radius: 6px;
                padding: 4px;
                color: {Theme.TEXT_PRIMARY};
            }}
            QMenu::item:selected {{
                background: rgba(106, 90, 205, 0.2);
                border-radius: 4px;
            }}
        """)

        # Basic actions - ALWAYS available regardless of settings
        menu.addAction("Apri con browser", lambda: webbrowser.open(self.video.url))
        menu.addAction("Apri con VLC", self.play_with_vlc_forced)  # Forced VLC (ignores settings)
        menu.addAction("Aggiungi alla coda", self.add_to_queue)
        menu.addAction("Copia link", lambda: QApplication.clipboard().setText(self.video.url))
        menu.addAction("Scarica video", self.download_video)
        menu.addAction("Condividi su Mastodon", self.share_on_mastodon)

        # RSS feed actions for PeerTube videos
        if hasattr(self.video, 'author') and self.video.author and getattr(self.video, 'platform', '') == "PeerTube":
            menu.addAction("Aggiungi a RSS Feed", self.add_to_rss_feed)

        # Playlist actions
        if self.main_window:
            playlist_menu = menu.addMenu("Aggiungi alla playlist")
            for pl in PlaylistService.get_playlists():
                if pl.name != "Guarda dopo":
                    action = playlist_menu.addAction(pl.name)
                    action.triggered.connect(lambda _, p=pl.id: self.add_to_playlist(p))
            playlist_menu.addAction("Crea nuova playlist", self.create_new_playlist)
            
            if self.playlist_id is None:
                menu.addAction("Salva in Guarda dopo", self.toggle_watch_later)
            else:
                menu.addAction("Rimuovi dalla playlist", self.remove_from_current_playlist)

        menu.exec_(self.mapToGlobal(pos))

    def add_to_queue(self):
        """Aggiungi video alla coda di riproduzione."""
        # Add video to the playback queue.
        logger.info(f"Aggiunta del video alla coda: {self.video.title}")
        # Adding video to queue: {self.video.title}
        self.add_to_queue_requested.emit(self.video)

    def download_video(self):
        """Scarica il video con tracciamento del progresso."""
        # Download the video with progress tracking.
        download_dir = QFileDialog.getExistingDirectory(
            self,
            "Seleziona la cartella di destinazione",
            # Select destination folder
            str(Path.home())
        )
        if not download_dir:
            return

        progress_dialog = QProgressDialog(
            f"Scaricamento: {self.video.title}",
            # Downloading: {self.video.title}
            "Annulla",
            # Cancel
            0, 100, self
        )
        progress_dialog.setWindowTitle("Download in corso")
        # Download in progress
        progress_dialog.setWindowModality(Qt.WindowModal)
        progress_dialog.setAutoClose(False)
        progress_dialog.setValue(0)

        def update_progress(progress: int):
            progress_dialog.setValue(progress)

        def on_completed(filename: str, message: str):
            if not progress_dialog.wasCanceled():
                progress_dialog.setValue(100)
                QMessageBox.information(self, "Successo", f"Video salvato: {filename}")
                # Success, Video saved: {filename}
            progress_dialog.close()

        def on_error(url: str, error: str):
            if not progress_dialog.wasCanceled():
                QMessageBox.warning(self, "Errore", f"Download fallito: {error}")
                # Error, Download failed: {error}
            progress_dialog.close()

        worker = DownloadWorker(self.video.url, download_dir)
        worker.signals.progress.connect(update_progress)
        worker.signals.finished.connect(on_completed)
        worker.signals.error.connect(on_error)
        progress_dialog.canceled.connect(worker.stop)

        self.thread_pool.start(worker)
        progress_dialog.exec_()

    def share_on_mastodon(self):
        """Condividi video su Mastodon."""
        # Share video on Mastodon.
        instance, ok = QInputDialog.getText(
            self,
            "Condividi su Mastodon",
            # Share on Mastodon
            "Inserisci l'URL della tua istanza Mastodon (es. mastodon.social):"
            # Enter your Mastodon instance URL (e.g., mastodon.social):
        )
        if ok and instance:
            instance = instance.strip()
            if not instance.startswith(("http://", "https://")):
                instance = f"https://{instance}"
            share_url = f"{instance}/share?text={self.video.title} {self.video.url}"
            webbrowser.open(share_url)

    def add_to_rss_feed(self):
        """Aggiungi il video a un feed RSS."""
        # Add the video to an RSS feed.
        logger.info(f"Tentativo di aggiungere feed RSS per il video: {self.video.title}")
        # Attempting to add RSS feed for video: {self.video.title}
        if not hasattr(self.video, 'url') or not self.video.url:
            logger.error("URL del video non disponibile.")
            # Video URL not available.
            QMessageBox.warning(self, "Errore", "URL del video non disponibile.")
            # Error, Video URL not available.
            return

        try:
            logger.info(f"Estrazione dell'ID del canale dall'URL del video: {self.video.url}")
            # Extracting channel ID from video URL: {self.video.url}
            channel_id = self.extract_channel_id(self.video.url)
            if not channel_id:
                logger.error("Impossibile estrarre l'ID del canale.")
                # Failed to extract channel ID.
                QMessageBox.warning(self, "Errore", "Impossibile estrarre l'ID del canale dal video.")
                # Error, Failed to extract channel ID from video.
                return

            feed_url = self.generate_rss_url(self.video.url, channel_id)
            logger.info(f"URL feed RSS generato: {feed_url}")
            # Generated RSS feed URL: {feed_url}

            existing_feeds = RSSService.get_feeds()
            existing_urls = [feed.url for feed in existing_feeds]
            logger.info(f"URL feed esistenti: {existing_urls}")
            # Existing feed URLs: {existing_urls}

            if feed_url in existing_urls:
                logger.warning("Il feed RSS esiste già.")
                # RSS feed already exists.
                QMessageBox.warning(self, "Attenzione", "Questo feed RSS è già stato aggiunto.")
                # Warning, This RSS feed has already been added.
                return

            logger.info("Aggiunta del feed RSS al database...")
            # Adding RSS feed to database...
            if RSSService.add_feed(feed_url):
                logger.info("Feed RSS aggiunto con successo!")
                # RSS feed added successfully!
                QMessageBox.information(
                    self,
                    "Successo",
                    f"Feed RSS del canale '{self.video.author}' aggiunto con successo!"
                    # Success, Channel '{self.video.author}' RSS feed added successfully!
                )
                if self.main_window and hasattr(self.main_window, 'statusBar'):
                    QTimer.singleShot(0, lambda: self.main_window.statusBar().showMessage(
                        f"✓ Feed RSS del canale '{self.video.author}' aggiunto."
                        # ✓ Channel '{self.video.author}' RSS feed added.
                    ))
            else:
                logger.error("Impossibile aggiungere il feed RSS.")
                # Failed to add RSS feed.
                QMessageBox.warning(self, "Errore", "Impossibile aggiungere il feed RSS del canale.")
                # Error, Failed to add channel RSS feed.
        except Exception as e:
            logger.error(f"Errore nell'aggiunta del feed RSS: {e}")
            # Error adding RSS feed: {e}
            QMessageBox.warning(
                self,
                "Errore",
                f"Errore nell'aggiunta del feed RSS: {str(e)}"
                # Error, Error adding RSS feed: {str(e)}
            )

    def extract_channel_id(self, video_url: str) -> Optional[str]:
        """Estrai l'ID del canale da un URL video PeerTube."""
        # Extract the channel ID from a PeerTube video URL.
        try:
            logger.info(f"Estrazione dell'ID del canale dall'URL: {video_url}")
            # Extracting channel ID from URL: {video_url}
            channel_id = self.extract_video_channel_id(video_url)
            if channel_id:
                logger.info(f"ID del canale estratto: {channel_id}")
                # Extracted channel ID: {channel_id}
                return channel_id

            # Soluzione alternativa: Prova a estrarre l'ID del canale dal link del feed RSS
            # Fallback: Try to extract channel ID from RSS feed link
            response = requests.get(video_url, timeout=10)
            soup = BeautifulSoup(response.text, 'html.parser')
            rss_link = soup.find('a', href=lambda href: href and '/feeds/videos.xml?videoChannelId=' in href)
            if rss_link:
                channel_id = rss_link['href'].split('videoChannelId=')[1]
                logger.info(f"ID del canale estratto dal link RSS: {channel_id}")
                # Extracted channel ID from RSS link: {channel_id}
                return channel_id

            logger.warning("Impossibile estrarre l'ID del canale.")
            # Failed to extract channel ID.
            return None
        except Exception as e:
            logger.error(f"Errore nell'estrazione dell'ID del canale: {e}")
            # Error extracting channel ID: {e}
            return None

    def extract_video_channel_id(self, video_url: str) -> Optional[str]:
        """Estrai l'ID del canale video da un URL video PeerTube."""
        # Extract the video channel ID from a PeerTube video URL.
        try:
            logger.info(f"Estrazione dell'ID del canale video dall'URL: {video_url}")
            # Extracting video channel ID from URL: {video_url}
            response = requests.get(video_url, timeout=10)
            response.raise_for_status()
            soup = BeautifulSoup(response.text, 'html.parser')

            # Prova a trovare l'ID del canale nei tag script
            # Try to find channel ID in script tags
            scripts = soup.find_all('script')
            for script in scripts:
                script_text = script.string
                if script_text and 'videoChannelId' in script_text:
                    match = re.search(r'videoChannelId[\'"\s:]=[\'"\s]*([^\'"\s]+)', script_text)
                    if match:
                        channel_id = match.group(1)
                        logger.info(f"Trovato videoChannelId nello script: {channel_id}")
                        # Found videoChannelId in script: {channel_id}
                        return channel_id

            # Prova a tre l'ID del canale negli attributi data
            # Try to find channel ID in data attributes
            elements_with_data = soup.find_all(attrs={"data-video-channel-id": True})
            if elements_with_data:
                channel_id = elements_with_data[0]['data-video-channel-id']
                logger.info(f"Trovato videoChannelId nell'attributo data: {channel_id}")
                # Found videoChannelId in data attribute: {channel_id}
                return channel_id

            # Prova a trovare l'ID del canale nel link del feed RSS
            # Try to find channel ID in RSS feed link
            rss_link = soup.find('a', href=lambda href: href and '/feeds/videos.xml?videoChannelId=' in href)
            if rss_link:
                channel_id = rss_link['href'].split('videoChannelId=')[1]
                logger.info(f"Trovato videoChannelId nel link RSS: {channel_id}")
                # Found videoChannelId in RSS link: {channel_id}
                return channel_id

            # Prova a recuperare l'ID del canale dall'API di PeerTube
            # Try to fetch channel ID from PeerTube API
            video_uuid = video_url.split('/')[-1]
            api_url = f"https://{urlparse(video_url).netloc}/api/v1/videos/{video_uuid}"
            logger.info(f"Recupero URL API: {api_url}")
            # Fetching API URL: {api_url}
            api_response = requests.get(api_url, timeout=10)
            if api_response.status_code == 200:
                api_data = api_response.json()
                if 'channel' in api_data and 'id' in api_data['channel']:
                    channel_id = str(api_data['channel']['id'])
                    logger.info(f"Trovato videoChannelId nella risposta API: {channel_id}")
                    # Found videoChannelId in API response: {channel_id}
                    return channel_id

            logger.warning("Impossibile trovare videoChannelId con alcun metodo.")
            # Failed to find videoChannelId in any method.
            return None
        except Exception as e:
            logger.error(f"Errore nell'estrazione dell'ID del canale video: {e}")
            # Error extracting video channel ID: {e}
            return None

    def generate_rss_url(self, video_url: str, channel_id: str) -> str:
        """Genera l'URL del feed RSS per un canale PeerTube."""
        # Generate the RSS feed URL for a PeerTube channel.
        base_domain = f"{urlparse(video_url).scheme}://{urlparse(video_url).netloc}"
        return f"{base_domain}/feeds/videos.xml?videoChannelId={channel_id}"

    def open_rss_feed_folder(self):
        """Apri la cartella del feed RSS nell'esplora file."""
        # Open the RSS feed folder in the file explorer.
        rss_folder = os.path.join(os.path.expanduser("~"), "purpletube_rss")
        logger.info(f"Apertura della cartella del feed RSS: {rss_folder}")
        # Opening RSS feed folder: {rss_folder}
        if not os.path.exists(rss_folder):
            os.makedirs(rss_folder)
            logger.info(f"Creata cartella del feed RSS: {rss_folder}")
            # Created RSS feed folder: {rss_folder}

        try:
            if sys.platform == 'win32':
                os.startfile(rss_folder)
            elif sys.platform == 'darwin':
                subprocess.Popen(['open', rss_folder])
            else:
                subprocess.Popen(['xdg-open', rss_folder])
            logger.info("Cartella del feed RSS aperta nell'esplora file.")
            # Opened RSS feed folder in file explorer.
        except Exception as e:
            logger.error(f"Errore nell'apertura della cartella del feed RSS: {e}")
            # Error opening RSS feed folder: {e}
            QMessageBox.warning(self, "Errore", f"Impossibile aprire la cartella del feed RSS: {str(e)}")
            # Error, Failed to open RSS feed folder: {str(e)}

    def create_new_playlist(self):
        """Crea una nuova playlist."""
        # Create a new playlist.
        name, ok = QInputDialog.getText(self, "Nuova Playlist", "Inserisci il nome della playlist:")
        # New Playlist, Enter playlist name:
        if ok and name:
            try:
                pl = Playlist(name=name)
                playlist_id = self.playlist_service.save_playlist(pl)
                if self.playlist_service.add_to_playlist(playlist_id, self.video):
                    if self.main_window and hasattr(self.main_window, 'statusBar'):
                        QTimer.singleShot(0, lambda: self.main_window.statusBar().showMessage(f"✓ Video aggiunto alla nuova playlist '{name}'"))
                        # ✓ Video added to new playlist '{name}'
                else:
                    QMessageBox.warning(self, "Errore", "Impossibile aggiungere il video alla nuova playlist")
                    # Error, Failed to add video to the new playlist
            except Exception as e:
                logger.error(f"Errore nella creazione della nuova playlist: {e}")
                # Error creating new playlist: {e}
                QMessageBox.warning(self, "Errore", f"Impossibile creare la playlist: {str(e)}")
                # Error, Failed to create playlist: {str(e)}

    def add_to_playlist(self, playlist_id: int):
        """Aggiungi video a una playlist esistente."""
        # Add video to an existing playlist.
        if self.playlist_service.add_to_playlist(playlist_id, self.video):
            playlist_name = next(p.name for p in self.playlist_service.get_playlists() if p.id == playlist_id)
            if self.main_window and hasattr(self.main_window, 'statusBar'):
                QTimer.singleShot(0, lambda: self.main_window.statusBar().showMessage(f"✓ Video aggiunto a {playlist_name}"))
                # ✓ Video added to {playlist_name}
        else:
            QMessageBox.warning(self, "Attenzione", "Il video è già in questa playlist")
            # Warning, Video already in this playlist

    def toggle_watch_later(self):
        """Attiva/Disattiva il video nella playlist Guarda dopo."""
        # Toggle video in Watch Later playlist.
        wl_playlist = self.playlist_service.get_watch_later_playlist()
        if not wl_playlist:
            return

        existing_ids = {v.video_id for v in wl_playlist.videos}
        if self.video.video_id not in existing_ids:
            if self.playlist_service.add_to_playlist(wl_playlist.id, self.video):
                QMessageBox.information(self, "Successo", "Video aggiunto a Guarda dopo")
                # Success, Video added to Watch Later
        else:
            if QMessageBox.question(
                self, "Attenzione", "Il video è già in Guarda dopo. Rimuovere?",
                # Warning, Video already in Watch Later. Remove?
                QMessageBox.Yes | QMessageBox.No
            ) == QMessageBox.Yes:
                if self.playlist_service.remove_from_playlist(wl_playlist.id, self.video.video_id):
                    self.watch_later_removed.emit()

    def remove_from_current_playlist(self):
        """Rimuovi il video dalla playlist corrente."""
        # Remove video from the current playlist.
        if self.playlist_id is not None and self.playlist_service.remove_from_playlist(self.playlist_id, self.video.video_id):
            self.video_removed.emit(self.video.video_id)

    def keyPressEvent(self, event):
        """Gestisci le scorciatoie da tastiera."""
        # Handle keyboard shortcuts.
        if event.key() == Qt.Key_Enter or event.key() == Qt.Key_Return:
            self.play_requested.emit(self.video)
            if self.main_window and hasattr(self.main_window, 'show_vlc_player'):
                self.play_with_vlc()
        elif event.key() == Qt.Key_Q and event.modifiers() == Qt.ControlModifier:
            self.add_to_queue()
        elif event.key() == Qt.Key_C and event.modifiers() == Qt.ControlModifier:
            QApplication.clipboard().setText(self.video.url)
            if self.main_window and hasattr(self.main_window, 'statusBar'):
                QTimer.singleShot(0, lambda: self.main_window.statusBar().showMessage("📋 URL del video copiato negli appunti!"))
                # 📋 Video URL copied to clipboard!
        elif event.key() == Qt.Key_D and event.modifiers() == Qt.ControlModifier:
            self.download_video()
        elif event.key() == Qt.Key_W and self.playlist_id is None:
            self.toggle_watch_later()
        elif event.key() == Qt.Key_Delete and self.playlist_id is not None:
            self.remove_from_current_playlist()
        else:
            super().keyPressEvent(event)
