"""
Feed content panel for displaying RSS feed items with infinite scroll, VLC integration, and queue support.
"""
from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QScrollArea, QLabel, QGridLayout, QMessageBox
)
from PyQt5.QtCore import Qt, pyqtSignal, QTimer
from config.theme import Theme
from core.ui.components.video_card import VideoCard
from core.models import Video
import logging

logger = logging.getLogger(__name__)

class FeedContentPanel(QWidget):
    """Panel for displaying RSS feed content with infinite scroll, VLC integration, and queue support."""
    video_card_play_requested = pyqtSignal(Video)  # Signal for playing a video
    add_to_queue_requested = pyqtSignal(Video)    # Signal for adding to queue

    def __init__(self, main_window, rss_service=None):
        super().__init__()
        self.main_window = main_window
        self.rss_service = rss_service
        self.current_feed = None
        self.feed_items = []
        self.current_offset = 0
        self.is_loading = False
        self.init_ui()

    def init_ui(self):
        """Initialize UI elements with a grid layout for feed items."""
        layout = QVBoxLayout(self)
        layout.setContentsMargins(10, 10, 10, 10)
        layout.setSpacing(10)

        # Title
        self.title_label = QLabel("Feed Content")
        self.title_label.setStyleSheet(f"font-size: 18px; font-weight: bold; color: {Theme.TEXT_PRIMARY};")
        layout.addWidget(self.title_label)

        # Scroll area for video grid
        self.scroll_area = QScrollArea()
        self.scroll_area.setWidgetResizable(True)
        self.scroll_area.setStyleSheet(f"background: {Theme.BACKGROUND}; border: none;")

        self.scroll_content = QWidget()
        self.grid_layout = QGridLayout(self.scroll_content)
        self.grid_layout.setSpacing(16)
        self.grid_layout.setContentsMargins(10, 10, 10, 10)

        self.scroll_area.setWidget(self.scroll_content)
        layout.addWidget(self.scroll_area)

        # Connect scroll event for infinite scroll
        self.scroll_area.verticalScrollBar().valueChanged.connect(self.on_scroll)

        # Add a placeholder message
        self.placeholder_label = QLabel("Select a feed from the Feed Manager to view its content")
        self.placeholder_label.setStyleSheet(f"color: {Theme.TEXT_SECONDARY}; font-style: italic;")
        self.placeholder_label.setAlignment(Qt.AlignCenter)
        self.grid_layout.addWidget(self.placeholder_label, 0, 0, 1, -1)

    def load_feed_content(self, feed):
        """Load content for the selected feed in a grid layout."""
        self.current_feed = feed
        self.feed_items = []
        self.current_offset = 0
        logger.info(f"Loading feed content for: {feed.title}")

        # Clear existing content
        for i in reversed(range(self.grid_layout.count())):
            widget = self.grid_layout.itemAt(i).widget()
            if widget:
                widget.setParent(None)

        # Load initial feed items
        self.load_more_feed_items()

    def load_more_feed_items(self):
        """Load the next batch of feed items and display them in a grid."""
        if self.is_loading or not self.current_feed:
            return

        self.is_loading = True
        logger.info(f"Loading more feed items (offset: {self.current_offset})...")

        try:
            items = self.rss_service.get_feed_items(
                feed_id=self.current_feed.id,
                limit=50,
                offset=self.current_offset
            )

            if not items:
                logger.info("No more items to load.")
                self.is_loading = False
                return

            logger.info(f"Found {len(items)} items for feed {self.current_feed.id}")
            self.feed_items.extend(items)

            # Display new feed items in a grid (4 columns)
            columns = 3
            for idx, item in enumerate(items):
                try:
                    video = Video(
                        title=item.title,
                        url=item.url,
                        thumbnail_url=item.thumbnail_url,
                        duration=item.duration,
                        video_id=item.video_id,
                        platform=item.platform,
                        author=item.author,
                        published=item.published
                    )

                    card = VideoCard(video, main_window=self.main_window)
                    card.play_requested.connect(self.video_card_play_requested.emit)
                    card.add_to_queue_requested.connect(self.add_to_queue_requested.emit)

                    row = idx // columns
                    col = idx % columns
                    self.grid_layout.addWidget(card, row, col)

                except Exception as e:
                    logger.error(f"Error creating video card for item {item.title}: {e}")
                    error_label = QLabel(f"Error loading video: {item.title}")
                    error_label.setStyleSheet(f"color: {Theme.DANGER};")
                    self.grid_layout.addWidget(error_label, idx, 0)

            # Update title with video count
            self.title_label.setText(f"Feed: {self.current_feed.title} ({len(self.feed_items)} videos)")
            self.main_window.statusBar().showMessage(f"Loaded {len(self.feed_items)} videos from {self.current_feed.title}")

            # Increment offset for next batch
            self.current_offset += len(items)

        except Exception as e:
            logger.error(f"Error loading feed content: {e}")
            QMessageBox.warning(
                self,
                "Error",
                f"Failed to load feed content: {str(e)}"
            )
            error_label = QLabel(f"Error: {str(e)}")
            error_label.setStyleSheet(f"color: {Theme.DANGER}; font-weight: bold;")
            error_label.setAlignment(Qt.AlignCenter)
            self.grid_layout.addWidget(error_label, 0, 0, 1, -1)

        finally:
            self.is_loading = False

    def on_scroll(self, value):
        """Handle scroll events for infinite scroll."""
        scroll_bar = self.scroll_area.verticalScrollBar()
        if not self.is_loading and scroll_bar.value() >= scroll_bar.maximum() - 100:
            self.load_more_feed_items()

